<?php
// includes/tg_boot.php
if (session_status() === PHP_SESSION_NONE) session_start();

if (!defined('TG_BOT_TOKEN')) {
  // Optionnel : charge config si non chargé ailleurs
  $cfg = __DIR__ . '/config.php';
  if (file_exists($cfg)) require_once $cfg;
}

function tg_extract_user_from_init($initData, $bot_token) {
  if (!$initData) return null;
  parse_str($initData, $arr);

  // Télégram renvoie 'user' JSON
  if (empty($arr['user'])) return null;

  // Vérif HMAC (recommandée)
  if (!empty($arr['hash'])) {
    $hash = $arr['hash'];
    unset($arr['hash']);
    ksort($arr);
    $data_check_string = '';
    foreach ($arr as $k => $v) $data_check_string .= $k.'='.$v."\n";
    $data_check_string = rtrim($data_check_string, "\n");
    $secret_key = hash('sha256', $bot_token, true);
    $hmac = hash_hmac('sha256', $data_check_string, $secret_key);
    // Si tu veux bloquer si hash invalide, décommente :
    // if (!hash_equals($hmac, $hash)) return null;
  }

  $u = json_decode($arr['user'], true);
  if (!$u || empty($u['id'])) return null;

  return [
    'telegram_id' => (string)$u['id'],
    'username'    => $u['username']    ?? null,
    'first_name'  => $u['first_name']  ?? null,
    'last_name'   => $u['last_name']   ?? null,
    'photo_url'   => $u['photo_url']   ?? null,
  ];
}

/**
 * Démarre/rafraîchit la session et upsert l'utilisateur.
 * Retourne l'array user (telegram_id, username, etc.) ou null.
 */
function tg_boot(PDO $pdo) {
  // 1) Déjà connu en session ?
  if (!empty($_SESSION['telegram_id'])) {
    $telegram_id = (string)$_SESSION['telegram_id'];

    // rafraîchit le last_seen
    $st = $pdo->prepare("UPDATE telegram_users SET last_seen = NOW() WHERE telegram_id = ?");
    $st->execute([$telegram_id]);
    return [
      'telegram_id' => $telegram_id,
      'username'    => $_SESSION['username']   ?? null,
      'first_name'  => $_SESSION['first_name'] ?? null,
      'last_name'   => $_SESSION['last_name']  ?? null,
      'photo_url'   => $_SESSION['photo_url']  ?? null,
    ];
  }

  // 2) Sinon, essaie de récupérer initData (header, GET ?__tg=..., POST __tg)
  $init = $_SERVER['HTTP_X_TELEGRAM_INIT_DATA'] ?? ($_GET['__tg'] ?? ($_POST['__tg'] ?? ''));
  if (!$init) return null; // pas lancé depuis le WebApp

  $bot_token = defined('TG_BOT_TOKEN') ? TG_BOT_TOKEN : '';
  $user = tg_extract_user_from_init($init, $bot_token);
  if (!$user) return null;

  // 3) Upsert utilisateur
  $sql = "INSERT INTO telegram_users (telegram_id, username, first_name, last_name, photo_url, created_at, last_seen, is_banned)
          VALUES (:id, :username, :first, :last, :photo, NOW(), NOW(), 0)
          ON DUPLICATE KEY UPDATE
            username = VALUES(username),
            first_name = VALUES(first_name),
            last_name = VALUES(last_name),
            photo_url = VALUES(photo_url),
            last_seen = NOW()";
  $st = $pdo->prepare($sql);
  $st->execute([
    ':id'     => $user['telegram_id'],
    ':username'=> $user['username'],
    ':first'  => $user['first_name'],
    ':last'   => $user['last_name'],
    ':photo'  => $user['photo_url'],
  ]);

  // 4) Session
  $_SESSION['telegram_id'] = $user['telegram_id'];
  $_SESSION['username']    = $user['username'];
  $_SESSION['first_name']  = $user['first_name'];
  $_SESSION['last_name']   = $user['last_name'];
  $_SESSION['photo_url']   = $user['photo_url'];

  return $user;
}
