<?php
// includes/currency.php
if (session_status() === PHP_SESSION_NONE) {
  session_start();
}

/* --- Catalogue des devises --- */
$GLOBALS['MLZ_CURRENCIES'] = [
  'EUR' => ['symbol'=>'€', 'rate'=>1.00, 'dec'=>2, 'pos'=>'suffix', 'decsep'=>',', 'thousand'=>' '],
  'USD' => ['symbol'=>'$', 'rate'=>1.08, 'dec'=>0, 'pos'=>'prefix','decsep'=>'.', 'thousand'=>','],
  'THB' => ['symbol'=>'฿', 'rate'=>37.40,'dec'=>0, 'pos'=>'suffix', 'decsep'=>'.', 'thousand'=>','],
];

/* --- Accès code courant (session → cookie fallback) --- */
function mlz_get_currency_code(): string {
  // 1) session
  $code = strtoupper((string)($_SESSION['mlz_currency'] ?? ''));
  if ($code && isset($GLOBALS['MLZ_CURRENCIES'][$code])) return $code;

  // 2) cookie (et on resynchronise la session)
  $cookie = strtoupper((string)($_COOKIE['mlz_currency'] ?? ''));
  if ($cookie && isset($GLOBALS['MLZ_CURRENCIES'][$cookie])) {
    $_SESSION['mlz_currency'] = $cookie;
    return $cookie;
  }
  return 'EUR';
}

/* --- Compat: alias historique --- */
function mlz_current_currency(): string { return mlz_get_currency_code(); }

/* --- Config d'une devise --- */
function mlz_currency_config(?string $code=null): array {
  $c = strtoupper($code ?? mlz_get_currency_code());
  return $GLOBALS['MLZ_CURRENCIES'][$c] ?? $GLOBALS['MLZ_CURRENCIES']['EUR'];
}

/* --- BOOT pour le JS --- */
function mlz_currency_boot_array(): array {
  $code = mlz_get_currency_code();
  $cfg  = mlz_currency_config($code);
  return ['code'=>$code] + $cfg;
}
function mlz_currency_boot_json(): string {
  return json_encode(mlz_currency_boot_array(), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
}

/* --- Formatage d'un prix à partir d'un montant EUR --- */
function mlz_format_price(float $eurAmount): string {
  $cfg  = mlz_currency_config();
  $amount = $eurAmount * (float)$cfg['rate'];
  $txt = number_format($amount, (int)$cfg['dec'], $cfg['decsep'], $cfg['thousand']);
  return $cfg['pos']==='prefix' ? ($cfg['symbol'].$txt) : ($txt.' '.$cfg['symbol']);
}

/* --- Setter (session + cookie 1 an) --- */
function mlz_set_currency(string $code): bool {
  $code = strtoupper(trim($code));
  if (!isset($GLOBALS['MLZ_CURRENCIES'][$code])) return false;

  $_SESSION['mlz_currency'] = $code;

  // Cookie
  $secure = !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';
  setcookie('mlz_currency', $code, [
    'expires'  => time() + 31536000, // 1 an
    'path'     => '/',
    'secure'   => $secure,
    'httponly' => false, // JS n'en a pas besoin mais on reste permissif
    'samesite' => 'Lax',
  ]);
  return true;
}

/* --- Taux BTC→EUR (helper) --- */
if (!function_exists('mlz_btc_eur_rate')) {
  function mlz_btc_eur_rate(): float {
    if (isset($GLOBALS['BTC_EUR_RATE'])) return (float)$GLOBALS['BTC_EUR_RATE'];
    if (getenv('BTC_EUR_RATE')) return (float) getenv('BTC_EUR_RATE');
    try {
      global $pdo;
      if ($pdo instanceof PDO) {
        $q = $pdo->query("SELECT value FROM settings WHERE name='BTC_EUR_RATE' LIMIT 1");
        $v = $q ? $q->fetchColumn() : null;
        if ($v !== false && $v !== null) return (float)$v;
      }
    } catch (Throwable $e) {}
    return 60000.0; // fallback
  }
}