<?php
// includes/currency.php
if (session_status() === PHP_SESSION_NONE) {
  session_start();
}

/* =========================
   Config + helpers Taux FX
   ========================= */

// TTL du cache des taux (12h)
if (!defined('MLZ_RATES_TTL')) define('MLZ_RATES_TTL', 12 * 3600);

// Où stocker le cache
function mlz_rates_cache_path(): string {
  $dir = __DIR__ . '/../cache';
  if (!is_dir($dir)) {
    @mkdir($dir, 0755, true);
  }
  if (!is_dir($dir) || !is_writable($dir)) {
    // fallback /tmp si pas de droits
    return rtrim(sys_get_temp_dir(), '/').'/mlz_rates.json';
  }
  return $dir . '/rates.json';
}

function mlz_http_get(string $url): ?string {
  // cURL si dispo
  if (function_exists('curl_init')) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_CONNECTTIMEOUT => 5,
      CURLOPT_TIMEOUT        => 8,
      CURLOPT_SSL_VERIFYPEER => true,
      CURLOPT_USERAGENT      => 'MLZ-FX/1.0'
    ]);
    $out = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($out !== false && $code >= 200 && $code < 300) return $out;
  }
  // file_get_contents sinon
  $ctx = stream_context_create(['http'=>['timeout'=>8,'user_agent'=>'MLZ-FX/1.0']]);
  $out = @file_get_contents($url, false, $ctx);
  return ($out !== false) ? $out : null;
}

/**
 * Charge les taux EUR->(USD, THB, EUR) depuis cache, sinon Internet.
 * Retourne toujours un tableau avec au moins EUR=1.0 + fallback sûrs.
 */
function mlz_load_rates(): array {
  $cacheFile = mlz_rates_cache_path();
  $now = time();

  // 1) Cache valable ?
  if (is_file($cacheFile)) {
    $raw = @file_get_contents($cacheFile);
    $obj = $raw ? @json_decode($raw, true) : null;
    if (is_array($obj) && isset($obj['ts'], $obj['rates']) && ($now - (int)$obj['ts'] < MLZ_RATES_TTL)) {
      $r = (array)$obj['rates'];
      // ensure EUR=1
      $r['EUR'] = 1.0;
      return $r;
    }
  }

  // 2) API publique gratuite (base EUR)
  $api = 'https://api.exchangerate.host/latest?base=EUR&symbols=USD,THB';
  $json = mlz_http_get($api);
  $rates = [
    'EUR' => 1.0,
    // fallbacks safe si l’API tombe (ne surtout pas casser l’affichage)
    'USD' => 1.08,
    'THB' => 37.40,
  ];
  if ($json) {
    $o = @json_decode($json, true);
    if (isset($o['rates']) && is_array($o['rates'])) {
      $rates['USD'] = (float)($o['rates']['USD'] ?? $rates['USD']);
      $rates['THB'] = (float)($o['rates']['THB'] ?? $rates['THB']);
    }
  }

  // 3) Écrit le cache (best effort)
  @file_put_contents($cacheFile, json_encode(['ts'=>$now, 'rates'=>$rates]));

  return $rates;
}

/** Applique les taux chargés aux devises déclarées (ne change pas tes décimales/positions) */
function mlz_apply_rates_to_config(array $rates): void {
  foreach ($rates as $code => $rate) {
    if (isset($GLOBALS['MLZ_CURRENCIES'][$code])) {
      // On ne touche qu’au 'rate' pour ne jamais casser ton formatage
      $GLOBALS['MLZ_CURRENCIES'][$code]['rate'] = (float)$rate;
    }
  }
}

/* =========================
   Catalogue des devises
   ========================= */
$GLOBALS['MLZ_CURRENCIES'] = [
  'EUR' => ['symbol'=>'€', 'rate'=>1.00, 'dec'=>2, 'pos'=>'suffix', 'decsep'=>',', 'thousand'=>' '],
  'USD' => ['symbol'=>'$', 'rate'=>1.08, 'dec'=>0, 'pos'=>'prefix','decsep'=>'.', 'thousand'=>','],
  'THB' => ['symbol'=>'฿', 'rate'=>37.40,'dec'=>0, 'pos'=>'suffix', 'decsep'=>'.', 'thousand'=>','],
];

// Met à jour les 'rate' depuis cache/API (à l’inclusion du fichier)
try {
  mlz_apply_rates_to_config( mlz_load_rates() );
} catch (Throwable $___e) {
  // Silencieux : en cas de souci réseau, on garde tes valeurs par défaut.
}

/* =========================
   Accesseurs & formatage
   ========================= */

/* --- Accès code courant (session → cookie fallback) --- */
function mlz_get_currency_code(): string {
  // 1) session
  $code = strtoupper((string)($_SESSION['mlz_currency'] ?? ''));
  if ($code && isset($GLOBALS['MLZ_CURRENCIES'][$code])) return $code;

  // 2) cookie (et on resynchronise la session)
  $cookie = strtoupper((string)($_COOKIE['mlz_currency'] ?? ''));
  if ($cookie && isset($GLOBALS['MLZ_CURRENCIES'][$cookie])) {
    $_SESSION['mlz_currency'] = $cookie;
    return $cookie;
  }
  return 'EUR';
}

/* --- Compat: alias historique --- */
function mlz_current_currency(): string { return mlz_get_currency_code(); }

/* --- Config d'une devise --- */
function mlz_currency_config(?string $code=null): array {
  $c = strtoupper($code ?? mlz_get_currency_code());
  return $GLOBALS['MLZ_CURRENCIES'][$c] ?? $GLOBALS['MLZ_CURRENCIES']['EUR'];
}

/* --- BOOT pour le JS --- */
function mlz_currency_boot_array(): array {
  $code = mlz_get_currency_code();
  $cfg  = mlz_currency_config($code);
  // NB: $cfg['rate'] contient déjà le taux mis à jour par mlz_apply_rates_to_config()
  return ['code'=>$code] + $cfg;
}
function mlz_currency_boot_json(): string {
  return json_encode(mlz_currency_boot_array(), JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
}

/* --- Formatage d'un prix à partir d'un montant EUR --- */
function mlz_format_price(float $eurAmount): string {
  $cfg  = mlz_currency_config();
  $amount = $eurAmount * (float)$cfg['rate'];
  $txt = number_format($amount, (int)$cfg['dec'], $cfg['decsep'], $cfg['thousand']);
  return $cfg['pos']==='prefix' ? ($cfg['symbol'].$txt) : ($txt.' '.$cfg['symbol']);
}

/* --- Setter (session + cookie 1 an) --- */
function mlz_set_currency(string $code): bool {
  $code = strtoupper(trim($code));
  if (!isset($GLOBALS['MLZ_CURRENCIES'][$code])) return false;

  $_SESSION['mlz_currency'] = $code;

  // Cookie
  $secure = !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off';
  setcookie('mlz_currency', $code, [
    'expires'  => time() + 31536000, // 1 an
    'path'     => '/',
    'secure'   => $secure,
    'httponly' => false, // JS n'en a pas besoin mais on reste permissif
    'samesite' => 'Lax',
  ]);
  return true;
}

/* --- Taux BTC→EUR (helper) --- */
if (!function_exists('mlz_btc_eur_rate')) {
  function mlz_btc_eur_rate(): float {
    if (isset($GLOBALS['BTC_EUR_RATE'])) return (float)$GLOBALS['BTC_EUR_RATE'];
    if (getenv('BTC_EUR_RATE')) return (float) getenv('BTC_EUR_RATE');
    try {
      global $pdo;
      if ($pdo instanceof PDO) {
        $q = $pdo->query("SELECT value FROM settings WHERE name='BTC_EUR_RATE' LIMIT 1");
        $v = $q ? $q->fetchColumn() : null;
        if ($v !== false && $v !== null) return (float)$v;
      }
    } catch (Throwable $e) {}
    return 60000.0; // fallback
  }
}