<?php
require_once __DIR__.'/includes/db.php';
require_once __DIR__.'/includes/tg_boot_soft.php';

/* ========= Helpers Wallet ========= */
function findBalanceColumn(PDO $pdo): ?string {
    $stmt = $pdo->query("SHOW COLUMNS FROM telegram_users");
    $cols = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'Field');
    foreach (['balance_sats','wallet_sats','btc_sats','btc_balance_sats','balance_btc_sats'] as $c) {
        if (in_array($c, $cols, true)) return $c;
    }
    return null;
}
function eurPerBtc(): float {
    // 1 BTC = X EUR (CoinGecko simple)
    $r = @file_get_contents('https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=eur');
    if (!$r) return 0.0;
    $j = json_decode($r, true);
    if (!is_array($j) || empty($j['bitcoin']['eur'])) return 0.0;
    return (float)$j['bitcoin']['eur'];
}
function eurToSats(float $eur, float $eurPerBtc): int {
    if ($eurPerBtc <= 0) return 0;
    $btc  = $eur / $eurPerBtc;
    $sats = (int)round($btc * 100000000);
    return max(0, $sats);
}
function ensureWalletTxTable(PDO $pdo): void {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS wallet_tx (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
            telegram_id BIGINT NOT NULL,
            direction ENUM('debit','credit') NOT NULL,
            amount_sats BIGINT NOT NULL,
            eur_total DECIMAL(18,2) NOT NULL,
            eur_per_btc DECIMAL(18,2) NOT NULL,
            ref VARCHAR(64) DEFAULT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            KEY(telegram_id), KEY(created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");
}
function safeRollback(PDO $pdo): void {
    try { if ($pdo->inTransaction()) $pdo->rollBack(); } catch (Throwable $e) {}
}

/* ========= Helpers Telegram Orders (template) ========= */

// même template par défaut que dans telegram_orders.php
const ORDER_TG_DEFAULT_TEMPLATE = "🔥 Commande MeltLabz

{PRODUCT_LINES}{PROMO_LINE}👤 Client :
• @{CLIENT_USERNAME}
• Prénom : {CLIENT_FIRST_NAME}
• Nom : {CLIENT_LAST_NAME}
• ID Telegram : {CLIENT_ID}

🏠 Livraison :
• Noms & Prénoms : {SHIP_FULL_NAME}
• Adresse : {SHIP_ADDRESS}
• Ville : {SHIP_CITY}
• Code Postal : {SHIP_ZIP}
• Téléphone : {SHIP_PHONE}

💳 Paiement : {PAYMENT_LINE}

🧾 Total du panier : {TOTAL_CART} €
🧾 Total à payer : {TOTAL_PAY} €";

/**
 * S'assure que la table existe + récupère/cree la ligne de config #1
 */
function order_tg_get_settings(PDO $pdo): array {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS telegram_order_settings (
          id TINYINT UNSIGNED NOT NULL PRIMARY KEY,
          bot_token VARCHAR(255) NOT NULL DEFAULT '',
          chat_id   VARCHAR(64)  NOT NULL DEFAULT '',
          template  TEXT NOT NULL,
          is_active TINYINT(1) NOT NULL DEFAULT 1,
          created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
          updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");

    $st = $pdo->prepare("SELECT * FROM telegram_order_settings WHERE id = 1 LIMIT 1");
    $st->execute();
    $row = $st->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        $ins = $pdo->prepare("
            INSERT INTO telegram_order_settings (id, bot_token, chat_id, template, is_active)
            VALUES (1, '', '', :tpl, 1)
        ");
        $ins->execute([':tpl' => ORDER_TG_DEFAULT_TEMPLATE]);
        $st->execute();
        $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    return $row;
}

/**
 * Remplace les {VARIABLES} dans le template
 */
function order_tg_format_message(string $tpl, array $vars): string {
    $search  = [];
    $replace = [];
    foreach ($vars as $k => $v) {
        $search[]  = '{' . $k . '}';
        $replace[] = $v;
    }
    return str_replace($search, $replace, $tpl);
}

/* ========= Session ========= */
session_start();
$telegram_id = $_SESSION['telegram_id'] ?? null;
if (!$telegram_id) { die("Non identifié via Telegram."); }

/* --- Panier --- */
$stmt = $pdo->prepare("
    SELECT pa.produit_id, p.nom, pa.quantite, pp.prix
    FROM paniers pa
    JOIN produits p ON pa.produit_id = p.id
    JOIN produit_prix pp ON pp.produit_id = p.id AND pp.quantite = pa.quantite
    WHERE pa.telegram_id = ?
");
$stmt->execute([$telegram_id]);
$items = $stmt->fetchAll(PDO::FETCH_ASSOC);
if (!$items) {
    echo "<script>alert('Votre panier est vide.'); window.location.href='panier.php';</script>";
    exit;
}

/* --- Adresse --- */
$adresse = $pdo->prepare("SELECT * FROM adresses WHERE telegram_id = ?");
$adresse->execute([$telegram_id]);
$adresseData = $adresse->fetch(PDO::FETCH_ASSOC);
if (!$adresseData) {
    echo "<script>alert('Veuillez enregistrer votre adresse avant de commander.'); window.location.href='panier.php';</script>";
    exit;
}

/* --- Utilisateur --- */
$user = $pdo->prepare("SELECT * FROM telegram_users WHERE telegram_id = ?");
$user->execute([$telegram_id]);
$userData = $user->fetch(PDO::FETCH_ASSOC);

/* --- Contenu & total (pour DB + {PRODUCT_LINES}) --- */
$contenu      = "";
$total        = 0.0;
$productLines = "";

foreach ($items as $it) {
    $p = (float)$it['prix'];

    // lignes pour le message Telegram
    $productLines .= "📦 Produit : {$it['nom']}\n";
    $productLines .= "📏 Quantité : {$it['quantite']} G – " . number_format($p, 2, ',', ' ') . " €\n\n";

    // contenu pour la table commandes
    $contenu .= "{$it['nom']} ({$it['quantite']}g) - ".number_format($p, 2, ',', ' ')."€\n";
    $total   += $p;
}

/* --- Code promo éventuel (recalcul & validation) --- */
$promoCode     = $_SESSION['promo_code'] ?? null;
$promoDiscount = 0.0;
$promoLine     = ""; // pour {PROMO_LINE}

if ($promoCode) {
    try {
        $stPromo = $pdo->prepare("SELECT * FROM promo_codes WHERE code = ? AND is_active = 1");
        $stPromo->execute([$promoCode]);
        $promo = $stPromo->fetch(PDO::FETCH_ASSOC);

        if ($promo) {
            $appliesAll = (int)$promo['applies_to_all'] === 1;
            $minTotal   = $promo['min_order_total'];
            $dtype      = $promo['discount_type'];      // 'percent' ou 'amount'
            $dval       = (float)$promo['discount_value'];

            $cartTotal     = $total;
            $eligibleTotal = 0.0;

            // Produits ciblés si ce n'est pas "tout le catalogue"
            $allowedIds = [];
            if (!$appliesAll) {
                $stIds = $pdo->prepare("SELECT produit_id FROM promo_code_produits WHERE promo_id = ?");
                $stIds->execute([(int)$promo['id']]);
                $allowedIds = array_map('intval', array_column($stIds->fetchAll(PDO::FETCH_ASSOC), 'produit_id'));
            }

            foreach ($items as $it2) {
                $price = (float)$it2['prix'];
                $pid   = isset($it2['produit_id']) ? (int)$it2['produit_id'] : 0;

                if ($appliesAll || in_array($pid, $allowedIds, true)) {
                    $eligibleTotal += $price;
                }
            }

            // Minimum de commande
            if ($minTotal !== null && $minTotal !== '' && $cartTotal < (float)$minTotal) {
                // minimum non atteint : on ignore simplement le code
            } elseif ($eligibleTotal > 0 && $dval > 0) {
                // Calcul de la remise
                if ($dtype === 'percent') {
                    $promoDiscount = round($eligibleTotal * $dval / 100, 2);
                } else {
                    $promoDiscount = min($dval, $eligibleTotal);
                }

                if ($promoDiscount > $cartTotal) {
                    $promoDiscount = $cartTotal;
                }

                $total       = max(0, $cartTotal - $promoDiscount);
                $discountStr = number_format($promoDiscount, 2, ',', ' ');

                // pour le message Telegram
                $promoLine = "💸 Code promo {$promoCode} : -{$discountStr} €\n\n";

                // pour la DB
                $contenu .= "\nCode promo: {$promoCode} (-{$discountStr}€)\n";
            }
        }
    } catch (Throwable $e) {
        // en cas d'erreur, on ignore le code promo pour ne pas bloquer la commande
    }
}

/* --- Paiement via Wallet : imposé (toujours ON) & débit PARTIEL auto --- */
$paid_via_wallet     = false;   // pilote le message TG
$wallet_debited_sats = 0;       // debug/trace
$wallet_paid_eur     = 0.0;     // montant € réellement payé via wallet (2 décimales)

$balanceCol = findBalanceColumn($pdo);
if ($balanceCol) {
    $eur_rate  = eurPerBtc();                 // 1 BTC = X EUR
    $sats_need = eurToSats((float)$total, $eur_rate);

    if ($sats_need > 0 && $eur_rate > 0) {
        try {
            // 1) DDL hors transaction
            ensureWalletTxTable($pdo);

            // 2) Débit en transaction
            $pdo->beginTransaction();

            // Lock + solde avant
            $st = $pdo->prepare("SELECT {$balanceCol} FROM telegram_users WHERE telegram_id = ? FOR UPDATE");
            $st->execute([$telegram_id]);
            $before_sats = (int)($st->fetchColumn() ?: 0);

            // Débit PARTIEL : tout ce qui est dispo, jusqu'au besoin
            $to_debit = min($before_sats, $sats_need);

            if ($to_debit > 0) {
                // Débit solde
                $up = $pdo->prepare("UPDATE telegram_users SET {$balanceCol} = {$balanceCol} - ? WHERE telegram_id = ?");
                $up->execute([$to_debit, $telegram_id]);

                // Montant € correspondant
                $eur_paid = round(($to_debit / 100000000) * $eur_rate, 2);

                // Log TX
                $ins = $pdo->prepare("
                    INSERT INTO wallet_tx (telegram_id, direction, amount_sats, eur_total, eur_per_btc, ref)
                    VALUES (?, 'debit', ?, ?, ?, 'order')
                ");
                $ins->execute([$telegram_id, $to_debit, $eur_paid, $eur_rate]);

                // Commit
                $pdo->commit();

                // Relecture pour vérifier
                $chk = $pdo->prepare("SELECT {$balanceCol} FROM telegram_users WHERE telegram_id = ?");
                $chk->execute([$telegram_id]);
                $after_sats = (int)($chk->fetchColumn() ?: 0);

                if ($after_sats === $before_sats - $to_debit) {
                    $paid_via_wallet     = true;
                    $wallet_debited_sats = $to_debit;
                    $wallet_paid_eur     = $eur_paid;
                } else {
                    // fallback via wallet_tx récent
                    $ver = $pdo->prepare("SELECT id FROM wallet_tx
                                          WHERE telegram_id=? AND direction='debit' AND amount_sats=? AND ref='order'
                                          ORDER BY id DESC LIMIT 1");
                    $ver->execute([$telegram_id, $to_debit]);
                    if ($ver->fetchColumn()) {
                        $paid_via_wallet     = true;
                        $wallet_debited_sats = $to_debit;
                        $wallet_paid_eur     = $eur_paid;
                    }
                }
            } else {
                // solde 0 → pas de débit (commande passe quand même)
                safeRollback($pdo);
            }
        } catch (Throwable $e) {
            safeRollback($pdo);
            // @error_log("CHECKOUT wallet EX: ".$e->getMessage());
        }
    }
}

/* === Calcul du reste à payer et formatage === */
$remaining_eur = max(0.0, round($total - $wallet_paid_eur, 2));
$fmt_paid      = number_format($wallet_paid_eur, 2, ',', ' ');
$fmt_total     = number_format($total,          2, ',', ' ');
$fmt_rest      = number_format($remaining_eur,  2, ',', ' ');

/* === Préparation des variables pour le template Telegram === */

// Client
$clientUsername   = $userData['username']   ?? 'Inconnu';
$clientFirstName  = $userData['first_name'] ?? 'N/A';
$clientLastName   = $userData['last_name']  ?? 'N/A';

// Livraison
$nomAdr   = trim(($adresseData['nom'] ?? '') . ' ' . ($adresseData['prenom'] ?? ''));
$adresseL = trim($adresseData['adresse'] ?? '');
$ville    = trim($adresseData['ville'] ?? '');
$cp       = trim($adresseData['code_postal'] ?? ($adresseData['cp'] ?? ''));
$tel      = trim($adresseData['telephone'] ?? '');

$ship_full_name = $nomAdr   !== '' ? $nomAdr   : 'N/A';
$ship_address   = $adresseL !== '' ? $adresseL : 'N/A';
$ship_city      = $ville    !== '' ? $ville    : 'N/A';
$ship_zip       = $cp       !== '' ? $cp       : 'N/A';
$ship_phone     = $tel; // on laisse vide si non renseigné

// Paiement
if ($wallet_paid_eur > 0) {
    $paymentLine = "Ce client a payé {$fmt_paid} € via son Wallet.";
} else {
    $paymentLine = "Ce client n'a pas payé via son Wallet.";
}

/* === Lecture de la config Telegram + génération du message === */
$tgSettings = order_tg_get_settings($pdo);
$bot_token  = trim($tgSettings['bot_token'] ?? '');
$chat_id    = trim($tgSettings['chat_id'] ?? '');
$template   = (string)($tgSettings['template'] ?? ORDER_TG_DEFAULT_TEMPLATE);
$is_active  = !empty($tgSettings['is_active']);

$vars = [
    'PRODUCT_LINES'     => $productLines,
    'PROMO_LINE'        => $promoLine,
    'CLIENT_USERNAME'   => $clientUsername,
    'CLIENT_FIRST_NAME' => $clientFirstName,
    'CLIENT_LAST_NAME'  => $clientLastName,
    'CLIENT_ID'         => (string)$telegram_id,
    'SHIP_FULL_NAME'    => $ship_full_name,
    'SHIP_ADDRESS'      => $ship_address,
    'SHIP_CITY'         => $ship_city,
    'SHIP_ZIP'          => $ship_zip,
    'SHIP_PHONE'        => $ship_phone,
    'PAYMENT_LINE'      => $paymentLine,
    'TOTAL_CART'        => $fmt_total,
    'TOTAL_PAY'         => $fmt_rest,
];

$texte = order_tg_format_message($template, $vars);

/* --- Envoi Telegram (si config OK & active) --- */
if ($is_active && $bot_token !== '' && $chat_id !== '') {
    $url    = "https://api.telegram.org/bot$bot_token/sendMessage";
    $params = ['chat_id' => $chat_id, 'text' => $texte, 'parse_mode' => 'Markdown'];
    @file_get_contents($url.'?'.http_build_query($params));
}

/* --- Enregistrer la commande & vider le panier --- */
$stmt = $pdo->prepare("INSERT INTO commandes (telegram_id, contenu, total) VALUES (?, ?, ?)");
$stmt->execute([$telegram_id, $contenu, $total]);

$pdo->prepare("DELETE FROM paniers WHERE telegram_id = ?")->execute([$telegram_id]);

unset(
    $_SESSION['promo_code'],
    $_SESSION['promo_id'],
    $_SESSION['promo_discount'],
    $_SESSION['promo_total_after']
);

echo "<script>
  try { document.cookie = 'use_wallet=; Max-Age=0; path=/'; } catch(e){}
  alert('Commande envoyée avec succès !');
  window.location.href='index.php';
</script>";