<?php
// /bot/webhook.php — Webhook Telegram pour MeltLabz (basé sur ton script Python)
declare(strict_types=1);

require_once __DIR__ . '/../includes/db.php'; // doit définir $pdo (PDO)

/* ==========================
   Helpers DB config bot
   ========================== */

/**
 * Récupère (ou crée) le bot #1 avec les valeurs par défaut de ton script.
 *
 * @return array
 */
function tg_get_bot_or_create(PDO $pdo): array {
    $st = $pdo->prepare("SELECT * FROM telegram_bots WHERE id = 1 LIMIT 1");
    $st->execute();
    $bot = $st->fetch(PDO::FETCH_ASSOC);

    if (!$bot) {
        $sql = "INSERT INTO telegram_bots
                (id, title, token, bot_username, group_chat_id, webhook_url, gif_url, image_enabled, chat_url, contact_url, is_active)
                VALUES
                (1,'Senzu Asia Bot',
                 '8058105739:AAGYQ2goMQqS1KOTaHQ9e6zTIfcTDJv1MiA',
                 'MeltLabzBot',
                 -1001759914839,
                 'https://meltlabz.us/bot/webhook.php',
                 'https://meltlabz.us/assets/loaders.gif',
                 1,
                 'https://t.me/MeltLabz',
                 'https://t.me/MeltLabzUS',
                 0)";
        $pdo->exec($sql);
        $st->execute();
        $bot = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    return $bot;
}

/* ==========================
   Constantes de texte (comme ton script Python)
   ========================== */

$TEXT = [
    'fr' => "🔥 <b>Bienvenue chez MeltLabz</b> 🔥\n\n"
          . "🧊 Notre hash d’exception est extrait <b>uniquement à l’eau et aux glaçons</b>.\n"
          . "📦 <b>Expéditions discrètes</b> partout dans le monde (sauf Russie).\n"
          . "💳 <b>Paiement crypto uniquement</b>.\n"
          . "⏱ <b>Livraison 5–14 jours ouvrés</b>.\n\n"
          . "🤖 Tout est automatisé jusqu’à la commande. Utilisez les boutons ci-dessous.",
    'en' => "🔥 <b>Welcome to MeltLabz</b> 🔥\n\n"
          . "🧊 Our exceptional hash is extracted <b>exclusively using ice water</b>.\n"
          . "📦 <b>Discreet worldwide shipping</b> (except Russia).\n"
          . "💳 <b>Crypto payments only</b>.\n"
          . "⏱ <b>Delivery 5–14 business days</b>.\n\n"
          . "🤖 Everything is automated up to checkout. Use the buttons below.",
    'es' => "🔥 <b>Bienvenido a MeltLabz</b> 🔥\n\n"
          . "🧊 Nuestro hash se extrae <b>exclusivamente con agua y hielo</b>.\n"
          . "📦 <b>Envíos discretos</b> a todo el mundo (excepto Rusia).\n"
          . "💳 <b>Pagos solo con cripto</b>.\n"
          . "⏱ <b>Entrega 5–14 días hábiles</b>.\n\n"
          . "🤖 Todo está automatizado hasta el checkout. Usa los botones abajo.",
];

$LANG_MENU_TEXT = "🌐 Please select your language:";

/* ==========================
   Helpers API Telegram
   ========================== */

/**
 * Appel API Telegram générique
 */
function tg_api_call(string $token, string $method, array $params = []): array {
    if (trim($token) === '') {
        return ['ok' => false, 'description' => 'Token manquant'];
    }

    $url = 'https://api.telegram.org/bot' . rawurlencode($token) . '/' . $method;

    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => "Content-Type: application/x-www-form-urlencoded\r\n",
            'content' => http_build_query($params),
            'timeout' => 10,
        ],
    ];

    $ctx = stream_context_create($opts);
    $resp = @file_get_contents($url, false, $ctx);

    if ($resp === false) {
        return ['ok' => false, 'description' => 'HTTP error'];
    }

    $data = json_decode($resp, true);
    if (!is_array($data)) {
        return ['ok' => false, 'description' => 'Invalid JSON'];
    }

    return $data;
}

/**
 * Envoi d'un message texte
 */
function tg_send_message(string $token, int $chatId, string $text, array $extra = []): array {
    $params = array_merge([
        'chat_id' => $chatId,
        'text'    => $text,
    ], $extra);

    return tg_api_call($token, 'sendMessage', $params);
}

/**
 * Envoi d'une animation (GIF)
 */
function tg_send_animation(string $token, int $chatId, string $gifUrl, array $extra = []): array {
    if (trim($gifUrl) === '') {
        return ['ok' => false, 'description' => 'GIF URL empty'];
    }

    $params = array_merge([
        'chat_id'   => $chatId,
        'animation' => $gifUrl,
    ], $extra);

    return tg_api_call($token, 'sendAnimation', $params);
}

/**
 * Répondre à un callback query
 */
function tg_answer_callback(string $token, string $callbackId, string $text = '', bool $alert = false): array {
    return tg_api_call($token, 'answerCallbackQuery', [
        'callback_query_id' => $callbackId,
        'text'              => $text,
        'show_alert'        => $alert ? true : false,
    ]);
}

/**
 * Modifier le texte d'un message
 */
function tg_edit_message_text(string $token, int $chatId, int $messageId, string $text, array $extra = []): array {
    $params = array_merge([
        'chat_id'    => $chatId,
        'message_id' => $messageId,
        'text'       => $text,
    ], $extra);

    return tg_api_call($token, 'editMessageText', $params);
}

/**
 * Modifier uniquement le markup (inline keyboard) d'un message
 */
function tg_edit_message_reply_markup(string $token, int $chatId, int $messageId, array $replyMarkup): array {
    return tg_api_call($token, 'editMessageReplyMarkup', [
        'chat_id'      => $chatId,
        'message_id'   => $messageId,
        'reply_markup' => json_encode($replyMarkup),
    ]);
}

/* ==========================
   Helpers: claviers depuis la DB
   ========================== */

/**
 * Construit un inline_keyboard depuis telegram_bot_buttons pour location/lang
 *
 * @return array ['inline_keyboard' => [...]]
 */
function tg_build_keyboard(PDO $pdo, int $botId, string $location, string $lang): array {
    $st = $pdo->prepare("
        SELECT *
        FROM telegram_bot_buttons
        WHERE bot_id = :bot_id
          AND location = :loc
          AND lang = :lang
          AND is_active = 1
        ORDER BY row_index, col_index, id
    ");
    $st->execute([
        ':bot_id' => $botId,
        ':loc'    => $location,
        ':lang'   => $lang,
    ]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    if (!$rows) {
        return ['inline_keyboard' => []];
    }

    $byRow = [];
    foreach ($rows as $btn) {
        $rowIx = (int)$btn['row_index'];
        if (!isset($byRow[$rowIx])) {
            $byRow[$rowIx] = [];
        }

        $button = ['text' => $btn['label']];

        $type = $btn['action_type'];
        if ($type === 'webapp') {
            if (!empty($btn['webapp_url'])) {
                $button['web_app'] = ['url' => $btn['webapp_url']];
            }
        } elseif ($type === 'callback') {
            if (!empty($btn['callback_data'])) {
                $button['callback_data'] = $btn['callback_data'];
            }
        } else { // url
            if (!empty($btn['url'])) {
                $button['url'] = $btn['url'];
            }
        }

        $byRow[$rowIx][] = $button;
    }

    ksort($byRow);
    $inline = array_values($byRow); // on n'a plus besoin des clés row_index

    return ['inline_keyboard' => $inline];
}

/**
 * Clavier DM
 */
function tg_keyboard_dm(PDO $pdo, int $botId, string $lang): array {
    return tg_build_keyboard($pdo, $botId, 'dm', $lang);
}

/**
 * Clavier Groupe
 */
function tg_keyboard_group(PDO $pdo, int $botId, string $lang): array {
    return tg_build_keyboard($pdo, $botId, 'group', $lang);
}

/**
 * Inline keyboard "choix de langue"
 */
function tg_keyboard_lang_menu(): array {
    return [
        'inline_keyboard' => [
            [
                ['text' => "🇺🇸 English",  'callback_data' => 'lang_en'],
                ['text' => "🇫🇷 Français", 'callback_data' => 'lang_fr'],
                ['text' => "🇪🇸 Español",  'callback_data' => 'lang_es'],
            ],
        ],
    ];
}

/* ==========================
   Logique principale
   ========================== */

$raw = file_get_contents('php://input');
if ($raw === false || $raw === '') {
    http_response_code(200);
    echo 'OK';
    exit;
}

$update = json_decode($raw, true);
if (!is_array($update)) {
    http_response_code(200);
    echo 'OK';
    exit;
}

// $pdo vient de ../includes/db.php
/** @var PDO $pdo */
$bot = tg_get_bot_or_create($pdo);
$TOKEN          = (string)($bot['token'] ?? '');
$GIF_URL        = (string)($bot['gif_url'] ?? '');
$GROUP_CHAT_ID  = isset($bot['group_chat_id']) ? (int)$bot['group_chat_id'] : 0;
$IMAGE_ENABLED  = !empty($bot['image_enabled']);
$BOT_ID         = (int)$bot['id'];

if ($TOKEN === '') {
    http_response_code(200);
    echo 'NO_TOKEN';
    exit;
}

/* --------------------------
   Gestion des messages texte
   -------------------------- */
if (isset($update['message'])) {
    $msg    = $update['message'];
    $chatId = (int)$msg['chat']['id'];
    $text   = isset($msg['text']) ? trim((string)$msg['text']) : '';

    // Commande /start
    if ($text !== '' && strpos($text, '/start') === 0) {
        // optionnel : paramètre deep-link, ex: "/start go"
        // list($cmd, $arg) = explode(' ', $text, 2); // si tu veux l'utiliser

        // Envoi du GIF (si activé)
        if ($IMAGE_ENABLED && $GIF_URL !== '') {
            tg_send_animation($TOKEN, $chatId, $GIF_URL);
        }

        // Menu choix de langue
        $kb = tg_keyboard_lang_menu();
        tg_send_message(
            $TOKEN,
            $chatId,
            $LANG_MENU_TEXT,
            [
                'reply_markup'             => json_encode($kb),
                'disable_web_page_preview' => true,
            ]
        );

        http_response_code(200);
        echo 'OK';
        exit;
    }

    // /chatid → renvoie l'ID du chat
    if ($text === '/chatid') {
        tg_send_message($TOKEN, $chatId, (string)$chatId);
        http_response_code(200);
        echo 'OK';
        exit;
    }

    // /postwelcome_fr / _en / _es → poste dans le groupe (config GROUP_CHAT_ID)
    if (in_array($text, ['/postwelcome_fr', '/postwelcome_en', '/postwelcome_es'], true)) {
        $lang = 'en';
        if ($text === '/postwelcome_fr') {
            $lang = 'fr';
        } elseif ($text === '/postwelcome_es') {
            $lang = 'es';
        }

        if (!isset($TEXT[$lang])) {
            $lang = 'en';
        }

        $targetChat = $GROUP_CHAT_ID ?: $chatId; // fallback sur le chat courant si pas de group_chat_id

        if ($IMAGE_ENABLED && $GIF_URL !== '') {
            tg_send_animation($TOKEN, $targetChat, $GIF_URL);
        }

        $kb = tg_keyboard_group($pdo, $BOT_ID, $lang);
        tg_send_message(
            $TOKEN,
            $targetChat,
            $TEXT[$lang],
            [
                'parse_mode'               => 'HTML',
                'reply_markup'             => json_encode($kb),
                'disable_web_page_preview' => true,
            ]
        );

        http_response_code(200);
        echo 'OK';
        exit;
    }

    // Autres messages → pour l'instant, on ne fait rien
    http_response_code(200);
    echo 'OK';
    exit;
}

/* --------------------------
   Gestion des callback_query
   -------------------------- */
if (isset($update['callback_query'])) {
    $cb = $update['callback_query'];
    $data = (string)($cb['data'] ?? '');
    $cbId = (string)($cb['id'] ?? '');

    $msg = $cb['message'] ?? null;
    if (!$msg) {
        tg_answer_callback($TOKEN, $cbId, '');
        http_response_code(200);
        echo 'OK';
        exit;
    }

    $chatId    = (int)$msg['chat']['id'];
    $messageId = (int)$msg['message_id'];

    // "lang_menu" → ré-affiche juste le menu de langues
    if ($data === 'lang_menu') {
        $kb = tg_keyboard_lang_menu();
        tg_edit_message_reply_markup($TOKEN, $chatId, $messageId, $kb);
        tg_answer_callback($TOKEN, $cbId, '');
        http_response_code(200);
        echo 'OK';
        exit;
    }

    // "lang_xx" → change la langue et affiche le texte + clavier DM
    if (strpos($data, 'lang_') === 0) {
        $lang = substr($data, 5);
        if (!in_array($lang, ['fr','en','es'], true)) {
            $lang = 'en';
        }
        global $TEXT;
        if (!isset($TEXT[$lang])) {
            $lang = 'en';
        }

        $kb = tg_keyboard_dm($pdo, $BOT_ID, $lang);

        tg_edit_message_text(
            $TOKEN,
            $chatId,
            $messageId,
            $TEXT[$lang],
            [
                'parse_mode'               => 'HTML',
                'reply_markup'             => json_encode($kb),
                'disable_web_page_preview' => true,
            ]
        );
        tg_answer_callback($TOKEN, $cbId, 'Language set ✅');

        http_response_code(200);
        echo 'OK';
        exit;
    }

    // Callback inconnu → ignore
    tg_answer_callback($TOKEN, $cbId, '');
    http_response_code(200);
    echo 'OK';
    exit;
}

// Si ce n'est ni message ni callback_query, on ignore
http_response_code(200);
echo 'OK';
exit;