/* assets/js/script.js — SPA + Overlays + Panier + Adresse + Vérification
   Checkout via checkout.php en AJAX (modale sexy, pas d'alert()) */
// === Masquer "Commander" si le panier est vide =============================
(function(){
  const $  = (s, r=document) => r.querySelector(s);

  function cartHasItems(){
    const tbody = $('#cart-tbody');
    const hasRows = !!(tbody && tbody.querySelector('tr'));
    const emptyMsg = $('#cart-empty');
    const emptyVisible = emptyMsg && getComputedStyle(emptyMsg).display !== 'none';

    // fallback via total
    let total = 0;
    const totEl = $('#cart-total');
    if (totEl) {
      const txt = (totEl.textContent || '').replace(/[^\d,.\-]/g,'').replace(',', '.');
      total = parseFloat(txt) || 0;
    }
    return (hasRows || total > 0.00001) && !emptyVisible;
  }

  function applyCheckoutVisibility(){
    const btn = $('#cart-checkout');
    if (!btn) return;
    const show = cartHasItems();

    // masque/affiche complètement
    btn.style.display = show ? '' : 'none';

    // blocage dur si visible alors que vide
    if (!show) {
      btn.setAttribute('aria-disabled','true');
      if (!btn.dataset.href) btn.dataset.href = btn.getAttribute('href') || '';
      btn.setAttribute('href', '#');
    } else {
      btn.removeAttribute('aria-disabled');
      if (btn.dataset.href) btn.setAttribute('href', btn.dataset.href);
    }
  }

  // Empêche le clic si caché/désactivé
  document.addEventListener('click', (e)=>{
    const a = e.target.closest('#cart-checkout');
    if (!a) return;
    if (a.style.display === 'none' || a.getAttribute('aria-disabled') === 'true') {
      e.preventDefault();
      e.stopPropagation();
    }
  }, true);

  // Observe les changements
  function startObservers(){
    const tbody = $('#cart-tbody');
    const total = $('#cart-total');
    if (tbody) new MutationObserver(applyCheckoutVisibility).observe(tbody, { childList:true, subtree:true });
    if (total) new MutationObserver(applyCheckoutVisibility).observe(total, { childList:true, characterData:true, subtree:true });
  }

  // MAJ quand on clique +/−/supprimer
  function bindCartButtons(){
    const panel = $('#cart-panel');
    if (!panel) return;
    panel.addEventListener('click', (e)=>{
      if (e.target.closest('.btn-inc, .btn-dec, .btn-del')) {
        setTimeout(applyCheckoutVisibility, 60);
      }
    });
  }

  // MAJ à l’ouverture du panier
  function bindOpeners(){
    ['#dock-cart','[data-open="cart-modal"]','#open-cart','[href="#cart-modal"]'].forEach(sel=>{
      const el = $(sel);
      if (el) el.addEventListener('click', ()=> setTimeout(applyCheckoutVisibility, 40));
    });
  }

  // Init
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', ()=>{
      applyCheckoutVisibility(); startObservers(); bindCartButtons(); bindOpeners();
    });
  } else {
    applyCheckoutVisibility(); startObservers(); bindCartButtons(); bindOpeners();
  }
})();

// === BAN-GATE — charge un JS serveur qui alerte + redirige si banni =========
(function(){
  const GATE_URL = '/ban_gate.js.php';
  const POLL_MS  = 250;
  const POLL_MAX = 16;

  function getCookie(n){
    const m = document.cookie.match(new RegExp('(?:^|;\\s*)'+n.replace(/[.*+?^${}()|[\]\\]/g,'\\$&')+'=([^;]*)'));
    return m ? decodeURIComponent(m[1]) : null;
  }
  function getTgIdCandidate(){
    try { if (window.Telegram?.WebApp?.initDataUnsafe?.user?.id) return String(Telegram.WebApp.initDataUnsafe.user.id); } catch(_){}
    return getCookie('tg_id');
  }
  function persist(id){
    try { document.cookie = 'tg_id=' + encodeURIComponent(id) + ';path=/;max-age=31536000'; } catch(_){}
  }
  function injectGate(id){
    if (document.querySelector('script[data-ban-gate]')) {
      document.querySelectorAll('script[data-ban-gate]').forEach(s=>s.remove());
    }
    const s = document.createElement('script');
    s.src = GATE_URL + (id ? ('?tg_id=' + encodeURIComponent(id)) : '');
    s.async = true;
    s.defer = true;
    s.setAttribute('data-ban-gate','1');
    document.head.appendChild(s);
  }

  async function run(){
    let id = getTgIdCandidate();
    if (id) { persist(id); injectGate(id); return; }
    let left = POLL_MAX;
    const timer = setInterval(()=>{
      id = getTgIdCandidate();
      if (id || --left <= 0){
        clearInterval(timer);
        if (id) persist(id);
        injectGate(id || null);
      }
    }, POLL_MS);
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', run);
  } else { run(); }

  // Navigation AJAX (pushState/popstate)
  const _ps = history.pushState;
  history.pushState = function(s,t,u){ const r=_ps.apply(this,arguments); try{ run(); }catch(_){} return r; };
  window.addEventListener('popstate', run);
})();

(() => {
  const _fetch = window.fetch;
  window.fetch = function(url, opts) {
    opts = opts || {};
    if (!opts.credentials) opts.credentials = 'include';
    opts.headers = Object.assign({'X-Requested-With':'XMLHttpRequest'}, opts.headers||{});
    return _fetch(url, opts);
  };
})()

function T(key, fallback){ try { return (window.I18N && window.I18N[key]) || fallback || key; } catch(_){ return fallback || key; } }

;

/* ---------- Utils ---------- */
const API = 'api/index.php';
const euro = n => (Number(n)||0).toFixed(2).replace('.', ',') + ' €';
const q = s => document.querySelector(s);
const qa = s => Array.from(document.querySelectorAll(s));

/* ====== FORMATTER MONNAIE — utilise la devise courante bootée par PHP ====== */
function formatFromBoot(eur){
  const C = window.MLZ_CURRENCY_BOOT || {code:'EUR',symbol:'€',rate:1,dec:2,pos:'suffix',decsep:',',thousand:' '};
  const amount = Number(eur||0) * Number(C.rate || 1);
  const dec = Math.max(0, C.dec|0);
  const fixed = amount.toFixed(dec);
  let [i,d] = fixed.split('.');
  i = i.replace(/\B(?=(\d{3})+(?!\d))/g, C.thousand);
  const body = dec ? (i + C.decsep + d) : i;
  return C.pos === 'prefix' ? (C.symbol + body) : (body + ' ' + C.symbol);
}

let TG_ID = null;
let TG_INIT = '';
try {
  const tg = window.Telegram?.WebApp;
  TG_ID  = tg?.initDataUnsafe?.user?.id || null;
  TG_INIT = tg?.initData || '';
} catch(_){}

/* --- helper: sommes-nous sur la page catégories ? --- */
function isCatlistPage(){
  try {
    if (/(?:^|\/)category_list\.php(?:\?|$)/i.test(location.pathname)) return true;
    if (history.state && history.state.view === 'catlist') return true;
    if (document.querySelector('.category-list, #category-list, .products-grid, .grid-products')) return true;
  } catch(_) {}
  return false;
}

/* ---------- Back button (fallback) ---------- */
try {
  q('#back-btn')?.addEventListener('click', ()=>{
    if (isCatlistPage()) { location.assign('/index.php'); return; }
    if (history.length > 1) history.back();
    else location.assign('/index.php');
  });
} catch(e){ console.error('back-btn', e); }

/* ---------- Fish-eye dock ---------- */
try {
  const dock = q('#dock');
  if (dock) {
    const items = qa('.dock-item');
    const maxScale = 1.35, radius = 95;
    const apply = x => items.forEach(el=>{
      const r = el.getBoundingClientRect(), cx = r.left + r.width/2;
      const t = Math.max(0, 1 - Math.abs(x - cx)/radius);
      el.style.transform = `translateY(${(-8*t)}px) scale(${1 + (maxScale-1)*t})`;
    });
    const reset = () => items.forEach(el=> el.style.transform='');

    dock.addEventListener('mousemove', e => apply(e.clientX), {passive:true});
    dock.addEventListener('mouseleave', reset, {passive:true});
    dock.addEventListener('pointermove', e => { if (e.pointerType === 'touch' || e.buttons) apply(e.clientX); }, {passive:true});
    dock.addEventListener('touchstart', e => apply(e.touches[0].clientX), {passive:true});
    dock.addEventListener('touchmove',  e => apply(e.touches[0].clientX), {passive:true});
    dock.addEventListener('touchend',   reset, {passive:true});
    items.forEach(el=>{
      el.addEventListener('pointerdown', ()=> el.style.transform='translateY(-6px) scale(1.18)');
      ['pointerup','pointercancel','pointerleave'].forEach(ev=> el.addEventListener(ev, ()=> el.style.transform=''));
    });
  }
} catch(e){ console.error('dock', e); }

/* ---------- Telegram avatar ---------- */
try {
  const tg = window.Telegram?.WebApp;
  const url = tg?.initDataUnsafe?.user?.photo_url;
  const img = q('#tg-avatar'); const fallback = q('#dock-avatar .fallback');
  if (url && img) { img.src = url; img.onload = () => { img.style.display='block'; if(fallback) fallback.style.display='none'; }; }
} catch(e){ console.error('avatar', e); }

/* ---------- Overlay helpers ---------- */
function bindOverlay(triggerSel, modalId){
  const trigger = q(triggerSel), overlay = q('#'+modalId);
  if (!overlay) return;
  const panel = overlay.querySelector('.glass-panel');
  const open  = ()=>{ overlay.classList.add('open'); panel?.getBoundingClientRect(); };
  const close = ()=> overlay.classList.remove('open');

  trigger && trigger.addEventListener('click', open);
  overlay.querySelectorAll('[data-close="'+modalId+'"]').forEach(b=> b.addEventListener('click', close));
  overlay.addEventListener('click', e => { if (e.target === overlay) close(); });
  panel?.addEventListener('click', e=> e.stopPropagation());
  document.addEventListener('keydown', e=>{ if(e.key==='Escape' && overlay.classList.contains('open')) close(); });

  overlay._open = open; overlay._close = close;
}
function wireOverlay(modalId){
  const overlay = q('#'+modalId); if(!overlay) return;
  const panel = overlay.querySelector('.glass-panel');
  const close = ()=> overlay.classList.remove('open');
  overlay.querySelectorAll('[data-close="'+modalId+'"]').forEach(b=> b.addEventListener('click', (e)=>{ e.preventDefault(); close(); }));
  overlay.addEventListener('click', e => { if (e.target === overlay) close(); });
  panel?.addEventListener('click', e=> e.stopPropagation());
  document.addEventListener('keydown', e=>{ if(e.key==='Escape' && overlay.classList.contains('open')) close(); });
  overlay._close = close;
}
function openOverlay(id){ const o=q('#'+id); o?._open ? o._open() : o?.classList.add('open'); }
function closeOverlay(id){ const o=q('#'+id); o?._close ? o._close() : o?.classList.remove('open'); }

/* Bind overlays */
try {
  bindOverlay('#dock-cart','cart-modal');
  bindOverlay('#dock-verify','verify-modal');
  wireOverlay('verify-result-modal');
  wireOverlay('address-modal');
  wireOverlay('address-saved-modal');
  wireOverlay('order-modal');
  bindOverlay('#dock-btc','btc-modal');
  wireOverlay('btc-modal');
} catch(e){ console.error('overlays', e); }

/* ---------- API helper ---------- */
async function call(action, payload={}){
  const body = new FormData();
  body.append('action', action);
  if (TG_ID) body.append('telegram_id', TG_ID);
  Object.entries(payload).forEach(([k,v])=> body.append(k, v));
  const res = await fetch(API, {
    method:'POST',
    body,
    headers: TG_INIT ? { 'X-Telegram-Init-Data': TG_INIT } : undefined,
    credentials: 'include'
  });
  const json = await res.json().catch(()=> ({}));
  if (!res.ok || !json.ok) throw new Error(json.error || T('api.error','Erreur API'));
  return json;
}

/* ---------- Panier (CURRENCY-FRIENDLY) ---------- */
async function loadCart(){
  try{
    const data = await call('cart.get');
    const tbody = q('#cart-tbody'); 
    const badge = q('#cart-badge'); 
    const empty = q('#cart-empty'); 
    const total = q('#cart-total');
    if(!tbody) return;

    tbody.innerHTML = '';

    // Aucun item
    if(!data.items || data.items.length===0){
      empty && (empty.style.display='block');
      if (badge) { badge.style.display='none'; badge.textContent=''; }
      if (total){
        const t = 0;
        total.classList.add('js-price');
        total.dataset.priceEur = t.toFixed(2);
        total.dataset.totalEur = t.toFixed(2);
        total.textContent = formatFromBoot(t);
      }
      document.dispatchEvent(new Event('mlz:cart-refreshed'));
      return;
    }

    empty && (empty.style.display='none');

    // Lignes
    data.items.forEach(it=>{
      const lineEUR = Number(it.line_total || 0);
      const tr=document.createElement('tr');
      tr.innerHTML=`
        <td class="col-prod">${it.nom} <small>(${it.pack_qty}g)</small></td>
        <td class="col-qty">
          <div class="qty" data-pid="${it.product_id}" data-pack="${it.pack_qty}">
            <button class="qty-btn" data-act="dec" aria-label="Diminuer">−</button>
            <span class="q">${it.units}</span>
            <button class="qty-btn" data-act="inc" aria-label="Augmenter">+</button>
          </div>
        </td>
        <td class="col-price">
          <span class="p js-price" data-price-eur="${lineEUR.toFixed(2)}">
            ${formatFromBoot(lineEUR)}
          </span>
        </td>
        <td class="col-del">
          <button class="del-btn" data-pid="${it.product_id}" data-pack="${it.pack_qty}" aria-label="Supprimer">×</button>
        </td>`;
      tbody.appendChild(tr);
    });

    // Total
    if (total){
      const t = Number(data.total || 0);
      total.classList.add('js-price');
      total.dataset.priceEur = t.toFixed(2);
      total.dataset.totalEur = t.toFixed(2); // pour le module "wallet"
      total.textContent = formatFromBoot(t);
    }

    // Badge
    if (badge){
      const cnt = (data.count != null) ? data.count
                 : data.items.reduce((s,i)=>s+(i.units||0),0);
      if (cnt>0){ badge.style.display='grid'; badge.textContent = cnt; }
      else { badge.style.display='none'; badge.textContent=''; }
    }

    // Notifie autres modules (wallet, etc.)
    document.dispatchEvent(new Event('mlz:cart-refreshed'));
  }catch(e){ console.error('loadCart', e); }
}
q('#dock-cart')?.addEventListener('click', loadCart);

/* MAJ badge rapide */
function setCartBadge(count){
  const badge = q('#cart-badge');
  if (!badge) return;
  if (count && count>0){ badge.style.display='grid'; badge.textContent = count; }
  else { badge.style.display='none'; badge.textContent=''; }
}
async function updateCartBadge(countIfKnown){
  if (typeof countIfKnown === 'number') return setCartBadge(countIfKnown);
  try{
    const data = await call('cart.get');
    setCartBadge(data.count ?? data.items?.length ?? 0);
  }catch(_){}
}

/* +/- quantité & suppression ligne */
q('#cart-tbody')?.addEventListener('click', async e=>{
  const btn = e.target.closest('.qty-btn');
  const del = e.target.closest('.del-btn');
  try{
    if(btn){
      const wrap = btn.closest('.qty');
      const pid  = +wrap.dataset.pid;
      const pack = +wrap.dataset.pack;
      const delta = btn.dataset.act==='inc' ? 1 : -1;
      await call('cart.qty',{ product_id:pid, pack_qty:pack, delta });
      await loadCart();
    }
    if(del){
      const pid = +del.dataset.pid;
      const pack = +del.dataset.pack;
      await call('cart.remove',{ product_id:pid, pack_qty:pack });
      await loadCart();
    }
  }catch(err){ console.warn(err.message); }
});

/* ---------- Adresse ---------- */
function fillAddressForm(addr){
  const set = (sel, val='') => { const el = typeof sel==='string' ? q(sel) : sel; if(el) el.value = val || ''; };
  set('#a-nom',     addr?.nom);
  set('#a-prenom',  addr?.prenom);
  set('#a-tel',     addr?.telephone ?? addr?.tel);
  set('#a-adr',     addr?.adresse);
  set('#a-cp',      addr?.code_postal ?? addr?.cp);
  set('#a-ville',   addr?.ville);
  set('#a-country', addr?.country ?? addr?.pays ?? '');   // <<< NEW
}


/* Ouvrir "Modifier mon adresse" depuis panier (pré-remplir puis ouvrir) */
(function(){
  async function fetchAndFillAddress(){
    try{ const res = await call('address.get'); fillAddressForm(res.address); }
    catch(e){ console.warn('address.get', e.message); }
  }

  // Depuis le panier
  document.addEventListener('click', async function(e){
    const link = e.target.closest('#cart-edit-address');
    if (!link) return;
    e.preventDefault();
    e.stopPropagation();
    e.stopImmediatePropagation();
    await fetchAndFillAddress();
    try { closeOverlay('cart-modal'); } catch(_){}
    openOverlay('address-modal');
  }, { capture:true });

  // Depuis l’avatar (dock)
  const avatarBtn = document.getElementById('dock-avatar');
  if (avatarBtn){
    avatarBtn.addEventListener('click', async (e)=>{
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
      await fetchAndFillAddress();
      openOverlay('address-modal');
    }, { capture:true });
  }
})();

/* Enregistrer l’adresse -> modale succès custom */
q('#address-save')?.addEventListener('click', async ()=>{
  const payload = {
    nom:     q('#a-nom')?.value || '',
    prenom:  q('#a-prenom')?.value || '',
    tel:     q('#a-tel')?.value || '',
    adresse: q('#a-adr')?.value || '',
    cp:      q('#a-cp')?.value || '',
    ville:   q('#a-ville')?.value || '',
    country: q('#a-country')?.value || ''   // <<< NEW
  };
  try{
    const res = await call('address.save', payload);
    fillAddressForm(res.address || payload);

    // Ligne récap pour la modale "success"
    const meta = [
      q('#a-prenom')?.value, q('#a-nom')?.value,
      q('#a-adr')?.value,
      [q('#a-cp')?.value, q('#a-ville')?.value, q('#a-country')?.value].filter(Boolean).join(' ')
    ].filter(Boolean).join(' · ');
    const box = document.querySelector('#addr-saved-meta');
    if (box) { box.textContent = meta ? '📍 ' + meta : ''; box.style.display = meta ? 'inline-block' : 'none'; }

    try { closeOverlay('address-modal'); } catch(_){}
    const os = q('#order-status'), oh=q('#order-heading'), om=q('#order-msg'), omb=q('#order-meta');
    if (os) { os.className='status-icon is-honey'; os.textContent='★'; }
    if (oh) oh.textContent = T('address.saved.title','Adresse enregistrée');
    if (om) om.textContent = '✅ ' + T('address.saved.message','Votre adresse a été sauvegardée avec succès.');
    if (omb) { omb.style.display='none'; omb.innerHTML=''; }
    openOverlay('order-modal');
  }catch(e){
    const os = q('#order-status'), oh=q('#order-heading'), om=q('#order-msg'), omb=q('#order-meta');
    if (os) { os.className='status-icon is-error'; os.textContent='!'; }
    if (oh) oh.textContent = T('order.oops','Oups…');
    if (om) om.textContent = e.message || T('address.error','Adresse non enregistrée.');
    if (omb) { omb.style.display='none'; omb.innerHTML=''; }
    openOverlay('order-modal');
  }
});


/* ---------- Vérification produit ---------- */
function formatDate(dstr){
  if(!dstr) return '—';
  const d = new Date(dstr.replace(' ', 'T'));
  if (isNaN(d)) return dstr;
  return d.toLocaleDateString(undefined, { year:'numeric', month:'long', day:'numeric' });
}
q('#verify-submit')?.addEventListener('click', async ()=>{
  const code = q('#verify-code')?.value?.trim() || '';
  if(!code) return (q('#verify-result').textContent=T('verify.empty','Entre un code.'));
  try{
    const res = await call('verify.check', {code});
    const c = res.code || {};

    const img  = q('#vr-image');
    const wrap = q('#vr-thumb');
    let src = (c.image_path || '').trim();
    if (img && wrap){
      if (src){
        if (!/^https?:\/\//i.test(src) && !src.startsWith('/')) src = '/' + src.replace(/^\.?\//,'');
        img.src = src; img.alt = c.strain || 'Produit'; wrap.style.display = '';
      } else {
        img.removeAttribute('src'); wrap.style.display = 'none';
      }
    }

    const m = /\[([^\]]+)\]\s*(.*)/.exec(c.strain || '');
    const cat  = m ? m[1] : '';
    const name = m ? m[2] : (c.strain || 'Produit');
    q('#vr-name') && (q('#vr-name').textContent = name);
    const catBadge = q('#vr-cat-badge');
    if (catBadge){ cat ? (catBadge.textContent = cat, catBadge.style.display='inline-flex') : (catBadge.style.display='none'); }
    q('#vr-date')  && (q('#vr-date').textContent  = formatDate(c.created_at));
    q('#vr-count') && (q('#vr-count').textContent = (c.verification_count ?? 0));
    q('#vr-code')  && (q('#vr-code').textContent  = c.code || code);
    q('#vr-desc')  && (q('#vr-desc').textContent  = c.description || '—');

    closeOverlay('verify-modal');
    openOverlay('verify-result-modal');
  }catch(e){
    q('#verify-result').textContent = e.message;
  }
});

/* --- Boot: panier --- */
document.addEventListener('DOMContentLoaded', () => {
  loadCart().catch(()=>{});
});
document.addEventListener('visibilitychange', () => {
  if (!document.hidden) loadCart().catch(()=>{});
});

/* ===== Zoomburst catégories -> liste produits via AJAX ===== */
(function(){
  const tiles = document.querySelector('.tiles');
  if (!tiles) return;

  const allTiles = Array.from(tiles.querySelectorAll('.tile'));
  const cats = allTiles.filter(a => (a.getAttribute('href') || a.href || '').includes('categorie='));

  cats.forEach(a=>{
    a.addEventListener('click', (e)=>{
      if (e.metaKey || e.ctrlKey) return;
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();

      const url = a.getAttribute('href') || a.href || '';
      const params = new URLSearchParams(url.split('?')[1] || '');
      const cat = params.get('categorie') || '';
      if (!cat) return;

      const pic  = a.querySelector('.pic') || a;
      const r  = pic.getBoundingClientRect();
      const cx = r.left + r.width/2, cy = r.top + r.height/2;
      const tx = innerWidth/2 - cx,  ty = innerHeight/2 - cy;
      const target = Math.min(innerWidth, innerHeight) * 0.46;
      const base   = Math.max(r.width, r.height);
      const scale  = Math.max(1.25, Math.min(target / base, 2.2));
      const duration = 900;

      document.body.classList.add('cat-anim');

      allTiles.forEach(t=>{
        t.style.transitionDuration = duration+'ms';
        t.querySelector('.pic')?.style.setProperty('transition-duration', duration+'ms');
      });

      allTiles.forEach(t=>{
        if (t === a) {
          t.classList.add('zooming');
        } else {
          t.style.opacity = '0';
          t.style.pointerEvents = 'none';
          const p = t.querySelector('.pic');
          if (p){
            p.style.opacity='0';
            p.style.filter='blur(2px)';
            p.style.transform='scale(.94)';
          }
        }
      });

      pic.style.transform = `translate(${tx}px, ${ty}px) scale(${scale})`;

      setTimeout(()=>{
        const tiles = document.querySelector('.tiles');
        if (!tiles) return;
        fetch(`/category_list.php?categorie=${encodeURIComponent(cat)}`, { headers: { 'X-Requested-With':'XMLHttpRequest' }})
          .then(r => r.text())
          .then(html => {
            if (window.__homeTilesHTML == null) window.__homeTilesHTML = tiles.innerHTML;
            tiles.innerHTML = html;
            try{ history.pushState({view:'catlist', cat}, '', `/category_list.php?categorie=${encodeURIComponent(cat)}`); }catch(_){}
            if (window.setBackVisible) window.setBackVisible(true);
            document.body.classList.remove('cat-anim');
          })
          .catch(console.error);
      }, duration + 20);
    }, {capture:true, passive:false});
  });
})();

/* === Header back button wiring (SPA home/catlist) === */
(function(){
  function setBackVisible(v){
    try{
      var btn = document.getElementById('back-btn');
      if(!btn) return;
      btn.hidden = !v;
      document.body.classList.toggle('has-back', !!v);
    }catch(_){}
  }
  window.setBackVisible = setBackVisible;

  function wireHeaderBack(){
    var btn = document.getElementById('back-btn');
    if(!btn) return;
    btn.addEventListener('click', function(e){
      e.preventDefault(); e.stopPropagation();

      if (isCatlistPage()) { location.assign('/index.php'); return; }

      if (history.state && (history.state.view === 'detail' || history.state.view === 'catlist' || history.state === 'prod')){
        history.back();
        return;
      }
      var tiles = document.querySelector('.tiles');
      if (tiles && window.__homeTilesHTML != null){
        tiles.innerHTML = window.__homeTilesHTML;
        setBackVisible(false);
        history.replaceState({view:'home'}, '', '/index.php');
        return;
      }
      location.assign('/index.php');
    }, {capture:true});
  }

  if (document.readyState === 'loading'){
    document.addEventListener('DOMContentLoaded', function(){ wireHeaderBack(); setBackVisible(false); });
  } else {
    wireHeaderBack(); setBackVisible(false);
  }
})();

/* Back/forward pour catlist/home */
window.addEventListener('popstate', function(e){
  var st = e.state || {};
  var tiles = document.querySelector('.tiles');
  if (!tiles) return;
  if (st.view === 'catlist' && st.cat){
    fetch(`/category_list.php?categorie=${encodeURIComponent(st.cat)}`, { headers: { 'X-Requested-With':'XMLHttpRequest' }})
      .then(r=>r.text())
      .then(html=>{
        tiles.innerHTML = html;
        if (window.setBackVisible) window.setBackVisible(true);
      })
      .catch(console.error);
  } else if (st.view === 'home' || /(?:^|\/)index\.php(?:\?|$)/.test(location.pathname) || location.pathname.endsWith('/')) {
    if (window.__homeTilesHTML != null){
      tiles.innerHTML = window.__homeTilesHTML;
      if (window.setBackVisible) window.setBackVisible(false);
    } else {
      if (window.setBackVisible) window.setBackVisible(false);
    }
  }
});

/* ---------- Ajouter au panier ---------- */
async function addToCart(productId, packQty){
  const res = await call('cart.add', { product_id: productId, pack_qty: packQty });
  await updateCartBadge(res.count);
  if (q('#cart-modal')?.classList.contains('open')) { await loadCart(); }
  return res;
}
function openCartModal(){ openOverlay('cart-modal'); }
window.MLZ_CART_FN = 'addToCart';
window.addToCart = addToCart;
window.loadCart = loadCart;
window.updateCartBadge = updateCartBadge;
window.openCartModal = openCartModal;

/* ---------- Checkout via checkout.php (AJAX) ---------- */
async function placeOrderViaCheckoutPhp() {
  const res = await fetch('checkout.php', {
    method: 'POST',
    headers: { 'X-Requested-With': 'XMLHttpRequest' },
    credentials: 'include'
  });
  const text = await res.text().catch(()=> '');
  if (!res.ok) throw new Error(
  T('order.http_error','Commande impossible (HTTP {status})').replace('{status}', res.status));
  if (/Panier vide|Non identifié|Erreur/i.test(text)) {
    const clean = text.replace(/<script[^>]*>.*?<\/script>/gis,'').replace(/<[^>]+>/g,'').trim();
    throw new Error(clean || T('order.error_alt','Erreur lors de la commande.'));
  }
  return { ok:true };
}

try { wireOverlay('order-modal'); } catch(_) {}
document.addEventListener('click', async function (e) {
  const btn = e.target.closest('#cart-checkout');
  if (!btn) return;

  if (e.button !== 0 || e.metaKey || e.ctrlKey || e.shiftKey || e.altKey) return;
  e.preventDefault();
  e.stopPropagation();
  e.stopImmediatePropagation();

  const label = btn.textContent;
  btn.classList.add('is-busy');
  btn.setAttribute('aria-busy', 'true');
  btn.textContent = T('order.sending','Envoi…');

  const S = q('#order-status'), H = q('#order-heading'), M = q('#order-msg'), MB = q('#order-meta');
  function show(ok, msg, metaHtml){
    if (S) { S.className = 'status-icon ' + (ok ? 'is-success' : 'is-error'); S.textContent = ok ? '✓' : '!'; }
    if (H) H.textContent = ok ? T('order.title','Félicitations !') : T('order.oops','Oups…');
    if (M) M.textContent = msg;
    if (MB) { MB.style.display = metaHtml ? 'inline-block' : 'none'; MB.innerHTML = metaHtml || ''; }
    try { closeOverlay('cart-modal'); } catch(_) {}
    openOverlay('order-modal');
  }

  try{
    if (typeof ensureI18N === 'function') { await ensureI18N(); }
    const res = await placeOrderViaCheckoutPhp();
    show(true, T('order.message','Votre commande a été envoyée avec succès. Notre équipe vous contactera très vite pour la suite 🎉'), '');
    await loadCart().catch(()=>{});
    updateCartBadge(0);
  }catch(err){
    if (typeof ensureI18N === 'function') { await ensureI18N(); }
    show(false, err?.message || T('order.error','Une erreur est survenue pendant l’envoi. Merci de réessayer.'), '');
  }finally{
    btn.textContent = label;
    btn.classList.remove('is-busy');
    btn.removeAttribute('aria-busy');
  }
}, { capture:true });

/* === Auto-init du slider détail même quand la page est injectée en AJAX === */
(function(){
  function initViewer(viewer){
    if (!viewer || viewer.dataset.mlzReady === '1') return;
    viewer.dataset.mlzReady = '1';

    const root   = viewer.closest('.product-image') || document;
    const slides = Array.from(viewer.querySelectorAll('.mlz-slide'));
    const prev   = viewer.querySelector('.mlz-arrow.prev');
    const next   = viewer.querySelector('.mlz-arrow.next');
    const thumbsWrap = root.querySelector('#mlz-thumbs');
    const thumbs = thumbsWrap ? Array.from(thumbsWrap.querySelectorAll('.mlz-thumb')) : [];
    if (!slides.length) return;

    viewer.style.position = viewer.style.position || 'relative';
    viewer.style.zIndex = viewer.style.zIndex || '10';

    let idx = Math.max(0, slides.findIndex(s => s.classList.contains('is-active')));

    function activate(n){
      n = (n + slides.length) % slides.length;
      slides.forEach((s,i)=>{
        if(i===n){ s.classList.add('is-active'); }
        else{
          const v=s.querySelector('video'); if(v){ v.pause(); v.currentTime=0; }
          s.classList.remove('is-active');
        }
      });
      thumbs.forEach((t,i)=> t.classList.toggle('is-active', i===n));
      const v2 = slides[n].querySelector('video'); if(v2){ v2.play().catch(()=>{}); }
      idx = n;
    }
    function goPrev(e){ e && (e.preventDefault(), e.stopPropagation()); activate(idx-1); }
    function goNext(e){ e && (e.preventDefault(), e.stopPropagation()); activate(idx+1); }

    if (prev){ prev.addEventListener('click', goPrev, {passive:false}); }
    if (next){ next.addEventListener('click', goNext, {passive:false}); }

    thumbs.forEach(t=>{
      const vid = t.querySelector('video'); if (vid) vid.style.pointerEvents = 'none';
      t.addEventListener('click', (e)=>{
        e.preventDefault(); e.stopPropagation();
        const i = parseInt(t.dataset.index,10);
        if(!Number.isNaN(i)) activate(i);
      }, {passive:false});
    });

    let startX=null, dragging=false;
    viewer.addEventListener('pointerdown', e=>{ startX=e.clientX; dragging=true; }, {passive:true});
    window.addEventListener('pointerup', ()=>{ dragging=false; startX=null; }, {passive:true});
    viewer.addEventListener('pointermove', e=>{
      if(!dragging || startX===null) return;
      const dx = e.clientX - startX;
      if(Math.abs(dx)>60){ dx<0 ? goNext(e) : goPrev(e); dragging=false; startX=null; }
    }, {passive:true});

    activate(idx);
  }

  function scan(root=document){
    root.querySelectorAll('#mlz-viewer').forEach(initViewer);
  }

  if (document.readyState !== 'loading') scan();
  else document.addEventListener('DOMContentLoaded', () => scan());

  const obs = new MutationObserver(muts=>{
    for (const m of muts){
      m.addedNodes.forEach(n=>{
        if (n.nodeType !== 1) return;
        if (n.id === 'mlz-viewer') initViewer(n);
        else if (n.querySelector) n.querySelectorAll('#mlz-viewer').forEach(initViewer);
      });
    }
  });
  obs.observe(document.body, {childList:true, subtree:true});

  // expose optionnel
  window.scan = scan;
})();

/* ==== MLZ detail page: safe init (slider + "Afficher plus") ================= */
(function (w, d) {
  w.MLZ = w.MLZ || {};
  const NS = w.MLZ;

  function ready(fn){ if (d.readyState !== 'loading') fn(); else d.addEventListener('DOMContentLoaded', fn, {once:true}); }

  function initSlider(viewer){
    try{
      if (!viewer || viewer.dataset.mlzReady === '1') return;
      const container = viewer.closest('.product-image') || viewer.parentElement || viewer;
      const slides = Array.from(viewer.querySelectorAll('.mlz-slide'));
      if (!slides.length){ viewer.dataset.mlzReady = '1'; return; }

      const prev   = viewer.querySelector('.mlz-arrow.prev');
      const next   = viewer.querySelector('.mlz-arrow.next');
      const thumbsWrap = container.querySelector('#mlz-thumbs');
      const thumbs = thumbsWrap ? Array.from(thumbsWrap.querySelectorAll('.mlz-thumb')) : [];

      thumbs.forEach(t => { const v=t.querySelector('video'); if(v) v.style.pointerEvents='none'; });

      let idx = Math.max(0, slides.findIndex(s => s.classList.contains('is-active')));

      function activate(n){
        n = (n + slides.length) % slides.length;
        slides.forEach((s,i)=>{
          if(i===n){ s.classList.add('is-active'); }
          else{
            const v=s.querySelector('video'); if(v){ v.pause(); v.currentTime=0; }
            s.classList.remove('is-active');
          }
        });
        thumbs.forEach((t,i)=> t.classList.toggle('is-active', i===n));
        const v2 = slides[n].querySelector('video'); if(v2){ v2.play().catch(()=>{}); }
        idx = n;
      }

      function goPrev(e){ if(e){ e.preventDefault(); e.stopPropagation(); } activate(idx-1); }
      function goNext(e){ if(e){ e.preventDefault(); e.stopPropagation(); } activate(idx+1); }

      if (prev) prev.addEventListener('click', goPrev, {passive:false});
      if (next) next.addEventListener('click', goNext, {passive:false});
      thumbs.forEach((t)=>{
        t.addEventListener('click', (e)=>{
          e.preventDefault(); e.stopPropagation();
          const i = parseInt(t.dataset.index,10);
          if (!Number.isNaN(i)) activate(i);
        }, {passive:false});
      });

      let startX=null, dragging=false;
      viewer.addEventListener('pointerdown', e=>{ startX=e.clientX; dragging=true; }, {passive:true});
      w.addEventListener('pointerup', ()=>{ dragging=false; startX=null; }, {passive:true});
      viewer.addEventListener('pointermove', e=>{
        if(!dragging || startX===null) return;
        const dx = e.clientX - startX;
        if(Math.abs(dx)>60){ dx<0 ? goNext(e) : goPrev(e); dragging=false; startX=null; }
      }, {passive:true});

      activate(idx);
      viewer.dataset.mlzReady = '1';
    }catch(_e){}
  }

  function initDesc(box){
    try{
      if (!box || box.dataset.descReady === '1') return;
      const content = box.querySelector('.mlz-desc-content');
      const meta    = content && content.querySelector('.product-meta');
      const toggle  = box.querySelector('.mlz-desc-toggle');
      const btn     = box.querySelector('.mlz-desc-btn');
      if (!content || !meta || !toggle || !btn){ box.dataset.descReady='1'; return; }

      toggle.checked = false;
      requestAnimationFrame(()=>{
        const needMore = content.scrollHeight > content.clientHeight + 1;
        if (needMore){
          box.classList.add('has-more');
        }else{
          btn.style.display = 'none';
          content.style.maxHeight = 'none';
          content.style.webkitMaskImage = 'none';
          content.style.maskImage = 'none';
          meta.style.display = 'block';
          meta.style.webkitLineClamp = 'unset';
        }
        box.dataset.descReady = '1';
      });
    }catch(_e){}
  }

  function initDetail(root){
    const scope = root || d;
    scope.querySelectorAll('.mlz-apple.mlz-detail').forEach(page=>{
      page.querySelectorAll('#mlz-viewer').forEach(initSlider);
      page.querySelectorAll('.mlz-desc').forEach(initDesc);
    });
  }

  ready(()=>initDetail());

  const mount = d.querySelector('#app') || d.body;
  const mo = new MutationObserver(muts=>{
    for (const m of muts){
      for (const n of m.addedNodes){
        if (n.nodeType !== 1) continue;
        if (n.matches?.('.mlz-apple.mlz-detail')) initDetail(n);
        else if (n.querySelector) {
          const found = n.querySelector('.mlz-apple.mlz-detail');
          if (found) initDetail(n);
        }
      }
    }
  });
  mo.observe(mount, {childList:true, subtree:true});

})(window, document);

/* === Verrou anti-rapetissage du viewer lors de l'ouverture de la description === */
(function(){
  const page = document.querySelector('.mlz-apple.mlz-detail');
  if (!page) return;

  const viewer = page.querySelector('#mlz-viewer');
  const toggle = page.querySelector('.mlz-desc-toggle');

  if (!viewer || !toggle) return;

  function lockViewer(){
    const h = viewer.getBoundingClientRect().height;
    if (h > 0){
      viewer.style.setProperty('--viewer-lock', h + 'px');
      viewer.classList.add('is-locked');
    }
  }
  function unlockViewer(){
    viewer.classList.remove('is-locked');
    viewer.style.removeProperty('--viewer-lock');
  }

  toggle.addEventListener('change', ()=>{
    if (toggle.checked) {
      lockViewer();
      setTimeout(lockViewer, 200);
    } else {
      unlockViewer();
    }
  });

  let rto;
  window.addEventListener('resize', ()=>{
    if (!viewer.classList.contains('is-locked')) return;
    clearTimeout(rto);
    rto = setTimeout(()=>{ unlockViewer(); lockViewer(); }, 120);
  });
})();

/* ===== BTC Deposit modal controller — version blindée ===== */
(function(){
  const API_BASE  = '/api';
  const CONF_TARGET = 1;

  function getTid(){
    if (window.__MLZ_TID) return Number(window.__MLZ_TID);
    try { const id = window?.Telegram?.WebApp?.initDataUnsafe?.user?.id; if (id) return Number(id); } catch(_){}
    return null;
  }

  async function apiGet(path, params){
    const url = new URL(API_BASE + path, location.origin);
    if (params) Object.entries(params).forEach(([k,v]) => url.searchParams.set(k, v));
    const r = await fetch(url.toString(), { credentials:'same-origin' });
    if (!r.ok) throw new Error('API '+r.status);
    return r.json();
  }
  async function apiPost(path, data){
    const url = new URL(API_BASE + path, location.origin);
    const r = await fetch(url.toString(), {
      method:'POST',
      headers:{ 'Content-Type':'application/json' },
      body: JSON.stringify(data || {}),
      credentials:'same-origin'
    });
    if (!r.ok) throw new Error('API '+r.status);
    return r.json();
  }

  let $modal,$close,$wait,$empty,$info,$addr,$qr,$copy,$conf,$gen,$ref1,$ref2;
  let $balBTCs = [], $balEURs = [];
  function grabEls(){
    $modal = document.getElementById('btc-modal');
    if (!$modal) return false;
    $close = $modal.querySelector('[data-close="btc-modal"]');

    $wait  = document.getElementById('btc-wait');
    $empty = document.getElementById('btc-empty');
    $info  = document.getElementById('btc-info');

    $addr   = document.getElementById('btc-address');
    $qr     = document.getElementById('btc-qr');
    $copy   = document.getElementById('btc-copy');
    $conf   = document.getElementById('btc-conf');

    $gen  = document.getElementById('btc-gen');
    $ref1 = document.getElementById('btc-refresh');
    $ref2 = document.getElementById('btc-refresh2');

    $balBTCs = Array.from($modal.querySelectorAll('.btc-balance-btc'));
    $balEURs = Array.from($modal.querySelectorAll('.btc-balance-eur'));

    $close?.addEventListener('click', () => closeOverlay('btc-modal'));
    $modal?.addEventListener('click', (e)=>{ if (e.target === $modal) closeOverlay('btc-modal'); });

    $gen?.addEventListener('click', onGenerate);
    $ref1?.addEventListener('click', ()=>{ const tid=getTid(); if (tid) loadState(tid); });
    $ref2?.addEventListener('click', ()=>{ const tid=getTid(); if (tid) loadState(tid); });
    $copy?.addEventListener('click', async ()=>{ try{ await navigator.clipboard.writeText($addr.textContent); $copy.textContent='Copié ✓'; setTimeout(()=> $copy.textContent='Copier', 1200);}catch(_){ } });

    return true;
  }
  if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', grabEls);
  else grabEls();

  function show(which){
    if (!$wait || !$empty || !$info) return;
    $wait.style.display  = (which==='wait')  ? 'block' : 'none';
    $empty.style.display = (which==='empty') ? 'block' : 'none';
    $info.style.display  = (which==='info')  ? 'block' : 'none';
  }
  function drawQR(value){
    if (!window.QRious || !$qr) return;
    new QRious({ element:$qr, value, size:220 });
  }
  function fmtEUR(num){
    try { return new Intl.NumberFormat('fr-FR',{style:'currency',currency:'EUR'}).format(num); }
    catch(_) { return '€'+Number(num||0).toFixed(2); }
  }
  function setBalances(btc, eurInEUR){
    const btcTxt = (Number(btc)||0).toFixed(8);
    $balBTCs.forEach(el => el.textContent = btcTxt);

    const eurBase = Number(eurInEUR)||0;
    $balEURs.forEach(el => {
      el.classList.add('js-price');
      el.setAttribute('data-price-eur', eurBase.toFixed(2));
      const C = window.MLZ_CURRENCY_BOOT;
      if (C && typeof C.rate !== 'undefined') {
        const amount = eurBase * Number(C.rate || 1);
        const dec = Math.max(0, C.dec|0);
        const fixed = amount.toFixed(dec);
        let [i,d] = fixed.split('.');
        i = i.replace(/\B(?=(\d{3})+(?!\d))/g, C.thousand);
        const body = dec ? (i + C.decsep + d) : i;
        el.textContent = C.pos === 'prefix' ? (C.symbol + body) : (body + ' ' + C.symbol);
      } else {
        el.textContent = fmtEUR(eurBase);
      }
    });
    try { document.dispatchEvent(new Event('mlz:prices:set')); } catch(_){}
  }

  async function eurFromRate(btc){
    try{
      const r = await apiGet('/rates.php');
      if (!r.ok) return 0;
      return (Number(btc)||0) * (r.btc_eur || 0);
    }catch(_){ return 0; }
  }

  function render(address, balanceSats, preset = {}){
    if ($addr) $addr.textContent = address;
    drawQR('bitcoin:'+address);

    let btc = (preset.btc != null)
      ? Number(preset.btc)
      : (Number(balanceSats)||0)/1e8;

    setBalances(btc, null);

    if (preset.eur != null) {
      setBalances(btc, Number(preset.eur));
    } else {
      eurFromRate(btc).then(eur => setBalances(btc, eur));
    }

    if ($conf) $conf.textContent = String(CONF_TARGET);
    show('info');
  }

  async function loadState(tid){
    try{
      const res=await apiGet('/get_balance.php',{telegram_id:tid});
      if(!res.ok||!res.btc_address) show('empty');
      else {
        const sats = (res.balance_sats != null)
          ? Number(res.balance_sats)
          : Math.round((Number(res.balance_btc)||0) * 1e8);
        render(res.btc_address, sats, {
          btc: (res.balance_btc != null) ? Number(res.balance_btc) : null,
          eur: (res.balance_eur != null) ? Number(res.balance_eur) : null
        });
      }
    }catch(_){ show('empty'); }
  }

  async function onGenerate(){
    const tid=getTid(); if(!tid) return;
    try{ const r=await apiPost('/assign_next_address.php',{telegram_id:tid}); if(!r.ok||!r.address) throw 0; render(r.address,0);}catch(_){ alert("Impossible de générer l'adresse pour le moment."); }
  }

  const dockBtn = document.getElementById('dock-btc');
  if (dockBtn) {
    dockBtn.addEventListener('click', () => {
      setTimeout(() => {
        if (!grabEls()) return;
        show('wait');
        const tid = getTid();
        if (!tid) {
          const onReady = () => { document.removeEventListener('mlz:telegramReady', onReady); const now=getTid(); if(now) loadState(now); else show('empty'); };
          document.addEventListener('mlz:telegramReady', onReady, { once:true });
          setTimeout(()=>{ const now=getTid(); if(now) loadState(now); else show('empty'); }, 4000);
          return;
        }
        loadState(tid);
      }, 50);
    });
  }
})();

/* === MLZ: tri & farms (delegation + open class) =============================== */
(function(){
  function qsParam(name){
    const p = new URLSearchParams(location.search);
    return p.get(name) || '';
  }
  function reloadCatList(params){
    const cat  = qsParam('categorie') || 'all';
    const usp  = new URLSearchParams();
    usp.set('categorie', cat);
    if (params.sort) usp.set('sort', params.sort);
    if (params.farm) usp.set('farm', params.farm);
    const url = '/category_list.php?' + usp.toString();
    fetch(url, { headers:{ 'X-Requested-With':'XMLHttpRequest' }})
      .then(r=>r.text())
      .then(html=>{
        const tiles = document.querySelector('.tiles');
        if (tiles) tiles.innerHTML = html;
        try { history.pushState({view:'catlist', cat}, '', url); } catch(_) {}
      })
      .catch(err=>{ console.error(err); location.assign(url); });
  }

  document.addEventListener('click', function(e){
    const btn = e.target.closest('#category-list .mlz-dd-btn');
    if (btn){
      e.preventDefault(); e.stopPropagation();
      const dd = btn.closest('.mlz-dd');
      if (!dd) return;
      const bar = dd.parentElement;
      bar?.querySelectorAll('.mlz-dd.open').forEach(x=>{ if (x!==dd) x.classList.remove('open'); });
      dd.classList.toggle('open');
      return;
    }
    const s = e.target.closest('#category-list .mlz-dd-menu a[data-sort]');
    if (s){
      e.preventDefault(); e.stopPropagation();
      const sort = s.dataset.sort || '';
      const farm = qsParam('farm') || '';
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
      reloadCatList({ sort, farm });
      return;
    }
    const f = e.target.closest('#category-list .mlz-dd-menu a[data-farm]');
    if (f){
      e.preventDefault(); e.stopPropagation();
      const farm = f.dataset.farm || '';
      const sort = qsParam('sort') || '';
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
      reloadCatList({ sort, farm });
      return;
    }
    if (!e.target.closest('#category-list .mlz-dd')){
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
    }
  }, true);
  document.addEventListener('keydown', function(e){
    if (e.key==='Escape') document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
  });
})();

/* ===== MeltLabz PJAX bridge v3: swap .mlz-apple, ré-init, pas de doublons JS ===== */
(function () {
  if (window.__mlzPjaxBridgeOnce) return;
  window.__mlzPjaxBridgeOnce = true;

  const TARGETS = ['.mlz-apple', '.tiles', '#app', '#ajax-view'];

  function findTarget(doc) {
    for (const sel of TARGETS) {
      const node = doc.querySelector(sel);
      if (node) return node;
    }
    return null;
  }

  const loadedSrc = new Set(
    Array.from(document.scripts).map(s => s.src).filter(Boolean)
  );

  function executeScripts(root) {
    const scripts = Array.from(root.querySelectorAll('script'));
    for (const s of scripts) {
      if (s.src) {
        if (loadedSrc.has(s.src)) { s.remove(); continue; }
        loadedSrc.add(s.src);
        const ns = document.createElement('script');
        ns.src = s.src;
        if (s.type) ns.type = s.type;
        ['crossorigin','referrerpolicy','integrity'].forEach(a => {
          if (s.hasAttribute(a)) ns.setAttribute(a, s.getAttribute(a));
        });
        s.parentNode.replaceChild(ns, s);
      } else {
        const ns = document.createElement('script');
        if (s.type) ns.type = s.type;
        ns.textContent = s.textContent || '';
        s.parentNode.replaceChild(ns, s);
      }
    }
  }

  function runInits() {
    try { if (typeof window.scan === 'function') window.scan(); } catch (e) {}
    try { if (typeof window.initCart    === 'function') window.initCart(); } catch (e) {}
    try { if (typeof window.initProduct === 'function') window.initProduct(); } catch (e) {}
    try { if (typeof window.initDetail  === 'function') window.initDetail(); } catch (e) {}
    try { document.dispatchEvent(new Event('mlz:tiles-ready')); } catch (e) {}
  }

  function swapInto(html, href) {
    const doc = new DOMParser().parseFromString(html, 'text/html');
    const next = findTarget(doc) || doc.body;
    const host = findTarget(document);
    if (!host || !next) return false;

    host.replaceWith(next);
    executeScripts(next);

    requestAnimationFrame(() => {
      try { history.pushState({ view: 'detail', href }, '', href); } catch (_) {}
      runInits();
      requestAnimationFrame(runInits);
    });
    return true;
  }

  // >>> MODIFIÉ : go(href, opts) avec soft-nav (sans spinner)
  function go(href, opts){
    const soft = !!(opts && opts.soft);
    const url = href + (href.includes('?') ? '&' : '?') + 'ajax=1';
    if (!soft) document.body.classList.add('is-loading');
    return fetch(url, {
      headers: { 'X-Requested-With': 'XMLHttpRequest' },
      credentials: 'include'
    })
      .then(r => r.text())
      .then(html => {
        if (!swapInto(html, href)) {
          document.open(); document.write(html); document.close();
        }
      })
      .catch(() => { window.location.href = href; })
      .finally(() => { if (!soft) document.body.classList.remove('is-loading'); });
  }

  // >>> AJOUT : expose
  window.mlzPjaxGo = go;

  // >>> AJOUT : event global “soft”
  document.addEventListener('mlz:navigate', function(e){
    const href = e.detail && e.detail.href;
    if (!href) return;
    e.preventDefault?.();
    e.stopPropagation?.();
    go(href, { soft:true });
  }, false);

  const ROUTE_SEL = '.product-card.js-route, a.js-route[href]';

  document.addEventListener('click', function (ev) {
    const n = ev.target.closest(ROUTE_SEL);
    if (!n) return;
    if (ev.metaKey || ev.ctrlKey || ev.button === 1) return;

    const href = n.getAttribute('data-href') || n.getAttribute('href');
    if (!href) return;

    ev.preventDefault();
    ev.stopPropagation();
    if (ev.stopImmediatePropagation) ev.stopImmediatePropagation();

    // >>> AJOUT : support data-soft-nav sur les liens
    const soft = n.hasAttribute('data-soft-nav');
    go(href, { soft });
  }, true);

  window.addEventListener('popstate', function (e) {
    const href = (e.state && e.state.href) ? e.state.href : location.href;
    go(href, { soft:true });
  });
})();

/* ===== Libellé du sélecteur de devise (i18n + PJAX) ====================== */
(function () {
  // Nom affiché pour chaque code devise (tu peux i18n si tu veux)
  function currencyName(code) {
    switch ((code || 'EUR').toUpperCase()) {
      case 'USD': return 'Dollar ($)';
      case 'THB': return 'THB (฿)';
      default:    return 'Euro (€)';
    }
  }

  function setCurrencyBtnLabel() {
    var btn = document.querySelector('.mlz-dd-btn[data-dd="currency"]');
    if (!btn) return;

    // Code devise courant (fourni côté PHP via MLZ_CURRENCY_BOOT)
    var code   = (window.MLZ_CURRENCY_BOOT && window.MLZ_CURRENCY_BOOT.code) || 'EUR';
    // Libellé traduit (lit window.I18N si dispo) — FR: DEVISE, EN: CURRENCY, ES: MONEDA
    var prefix = T('currency.button', 'DEVISE');

    btn.textContent = prefix + ' : ' + currencyName(code);
  }

  // On s’assure de repasser APRÈS les autres handlers
  function schedule() { requestAnimationFrame(() => setTimeout(setCurrencyBtnLabel, 0)); }

  // 1) initial
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', schedule, { once: true });
  } else {
    schedule();
  }
  // 2) quand la devise change (ton code déclenche mlz:currency-changed)
  document.addEventListener('mlz:currency-changed', schedule);
  // 3) quand l’i18n est prêt / langue change / page injectée via PJAX
  document.addEventListener('i18n:ready', schedule);
  window.addEventListener('languagechange', schedule);
  document.addEventListener('mlz:tiles-ready', schedule);
})();


/* ===== Tracker (online + derniers visiteurs index) — PURE JS, no PHP ===== */
(function(){
  if (window.__mlzTrackerOnce) return; 
  window.__mlzTrackerOnce = true;

  function _nameOf(r){
    if (!r) return '—';
    return r.username ? ('@'+r.username)
         : (r.first_name ? r.first_name
         : (r.telegram_id ? ('#'+r.telegram_id) : '—'));
  }
  function _fmtTime(iso){
    if(!iso) return '';
    var d = new Date(String(iso).replace(' ','T'));
    if (isNaN(d)) return '';
    var dt = (Date.now()-d.getTime())/1000;
    if (dt < 60)    return T('tracker.just_now','just now');
    if (dt < 3600)  return Math.floor(dt/60)+'m';
    if (dt < 86400) return Math.floor(dt/3600)+'h';
    return d.toLocaleString();
  }

  async function trackerPing(page){
    try{
      var u = (window.Telegram && window.Telegram.WebApp && window.Telegram.WebApp.initDataUnsafe && window.Telegram.WebApp.initDataUnsafe.user) || {};
      await call('track.ping', {
        page: page || '',
        username: u.username || '',
        first_name: u.first_name || ''
      });
    }catch(e){ /* silencieux */ }
  }

  async function trackerRefresh(){
    try{
      var data = await call('track.fetch', { limit: 5 });

      var cnt = data.count_online || (data.online && data.online.length) || 0;
      var $cnt = document.getElementById('trk-count');
      if ($cnt) $cnt.textContent = cnt;

      var $online = document.getElementById('trk-online');
      if ($online){
        var list = (data.online||[]).map(function(r){
          return '<li><span class="trk-name">'+_nameOf(r)+'</span><span class="trk-time">'+_fmtTime(r.last_seen)+'</span></li>';
        });
        $online.innerHTML = list.length ? list.join('') : (
          '<li class="trk-empty">'+T('tracker.none','Nobody online.')+'</li>'
        );
      }

      var $recent = document.getElementById('trk-recent');
      if ($recent){
        var list2 = (data.recent_index||[]).map(function(r){
          return '<li><span class="trk-name">'+_nameOf(r)+'</span><span class="trk-time">'+_fmtTime(r.last_index_at)+'</span></li>';
        });
        $recent.innerHTML = list2.length ? list2.join('') : (
          '<li class="trk-empty">'+T('tracker.none_recent','No recent visits yet.')+'</li>'
        );
      }
    }catch(e){ /* silencieux */ }
  }

  function boot(){
    // On active sur la home (tiles) ou si une carte tracker existe
    var isHome  = !!document.querySelector('.tiles');
    var hasCard = !!document.getElementById('mlz-tracker');
    if (!isHome && !hasCard) return;

    trackerPing(isHome ? 'index' : '');
    trackerRefresh();

    clearInterval(window.__mlzTrkPing);
    clearInterval(window.__mlzTrkRefresh);
    window.__mlzTrkPing    = setInterval(function(){ trackerPing(''); }, 60*1000);
    window.__mlzTrkRefresh = setInterval(trackerRefresh, 30*1000);
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', boot);
  } else boot();

  // Si la page est injectée (PJAX)
  document.addEventListener('mlz:tiles-ready', function(){
    trackerPing('index'); 
    trackerRefresh();
  });
})();


/* ==== FIX détail (qty + add-to-cart) — sans rien changer à detail.js ==== */
(() => {
  if (window.__mlzDetailDomFixOnce) return;
  window.__mlzDetailDomFixOnce = true;

  function ensureDetailDOM(root = document) {
    const scope =
      root.querySelector('.mlz-apple.mlz-detail') ||
      document.querySelector('.mlz-apple.mlz-detail') ||
      document.querySelector('.mlz-apple');
    if (!scope) return;

    const urlId = new URLSearchParams(location.search).get('id');
    const pid = scope.getAttribute('data-id') || urlId || '';

    scope.querySelectorAll('.prices-row .badge-price').forEach(pill => {
      pill.classList.add('js-qty');
      if (!pill.hasAttribute('data-pid') && pid) {
        pill.setAttribute('data-pid', pid);
      }
      if (!pill.hasAttribute('data-qty')) {
        const txt = pill.textContent || '';
        let m = txt.match(/(\d[\d.,]*)\s*G+/i) || txt.match(/(\d[\d.,]*)/);
        if (m) {
          const qty = m[1].replace(',', '.').trim();
          if (qty) pill.setAttribute('data-qty', qty);
        }
      }
    });

    let btn = scope.querySelector('#mlz-add-btn, .btn-add-cart, [data-add-to-cart]');
    if (btn && !btn.id) btn.id = 'mlz-add-btn';
    if (btn) {
      const hasSel = !!scope.querySelector('.badge-price.js-qty.is-selected');
      btn.disabled = !hasSel;
    }
  }

  function reinitDetail() {
    ensureDetailDOM();
    if (typeof window.initDetailPage === 'function') {
      window.initDetailPage();
    }
  }

  document.addEventListener('DOMContentLoaded', reinitDetail);
  document.addEventListener('mlz:tiles-ready', reinitDetail);

  const mo = new MutationObserver(() => ensureDetailDOM());
  mo.observe(document.body, { childList: true, subtree: true });
})();