/* mlz-filters-ultra.js — front-only filters for category page
   - NO fetch, NO reload, NO style/layout changes
   - Strict scope: only inside #category-list
   - Works after AJAX injection (delegated listeners + MutationObserver)
*/
(function(){
  const doc = document;
  const norm = s => (s||'').toString().toLowerCase().trim();
  const log  = (...args)=>{ if (window.DEBUG_MLZ) console.log('[MLZ]', ...args); };

  // ---- ROOT / GRID / CARDS detection (robust)
  function getRoot(){
    const r = doc.getElementById('category-list');
    if (!r) log('root not found (#category-list)');
    return r;
  }
  function getGrid(root){
    if (!root) return null;
    // prefer explicit containers if they exist
    const sel = [
      '.grid-products', '.mlz-products', '.products-grid',
      '.mlz-wrap', '.cards-wrap', '.cards', '.row'
    ];
    for (const s of sel){
      const el = root.querySelector(s);
      if (el) return el;
    }
    // fallback: direct child containing many anchors
    return root;
  }
  function getCards(grid){
    if (!grid) return [];
    // most specific first
    let nodes = Array.from(grid.querySelectorAll('a.product-card'));
    if (nodes.length) return nodes;
    // anchors to product.php (common)
    nodes = Array.from(grid.querySelectorAll('a[href*="product.php"]'));
    if (nodes.length) return nodes;
    // fallback: classic card classes
    nodes = Array.from(grid.querySelectorAll('.product-card, .card.product, .card-prod, .card'));
    return nodes;
  }

  // ---- PRICE extraction (min € inside the card)
  function extractMinPrice(card){
    // 1) data-price attr if present
    const dp = card.getAttribute('data-price');
    if (dp && !isNaN(parseFloat(dp.replace(',','.')))){
      return parseFloat(dp.replace(',','.'));
    }
    // 2) specific price elements
    const priceNode = card.querySelector('.badge-price, .price, [data-price]');
    if (priceNode){
      const t = (priceNode.getAttribute('data-price') || priceNode.textContent || '').trim();
      const n = parsePrice(t);
      if (isFinite(n)) return n;
    }
    // 3) generic scan
    const txt = (card.textContent || '').replace(/\s+/g,' ');
    const n = parsePrice(txt);
    return isFinite(n) ? n : 9e9;
  }
  function parsePrice(txt){
    // capture 1..n euros like "1 234,56 €" or "123,45€" or "12 €"
    const re  = /([0-9]{1,3}(?:[ .][0-9]{3})*|[0-9]+)(?:,([0-9]{1,2}))?\s*€/g;
    let m, best = Infinity;
    while ((m = re.exec(txt))){
      const n = parseFloat((m[1]||'0').replace(/[ .]/g,'') + '.' + (m[2]||'00'));
      if (!isNaN(n)) best = Math.min(best, n);
    }
    return best;
  }

  // ---- FARM extraction from card
  function extractFarm(card){
    // 1) data-farm on card
    const ds = card.getAttribute('data-farm');
    if (ds) return norm(ds);
    // 2) element with explicit farm info
    const el = card.querySelector('[data-farm], .farm-badge, .badge.farm, .badge[data-farm], .tag.farm');
    if (el){
      const attr = el.getAttribute('data-farm');
      return norm(attr || el.textContent);
    }
    // 3) fallback: none
    return '';
  }

  // ---- APPLY filter/sort on current DOM
  function applyFilters({sort, farm}){
    const root = getRoot(); if (!root) return;
    const grid = getGrid(root); if (!grid) return;
    const cards = getCards(grid);
    if (!cards.length){ log('no cards found'); return; }

    // filter
    const want = norm(farm||'');
    cards.forEach(c => {
      const cf = extractFarm(c);
      c.style.display = want ? (cf === want ? '' : 'none') : '';
    });

    // sort
    if (sort === 'price_asc' || sort === 'price_desc'){
      const vis = cards.filter(c => c.style.display !== 'none');
      vis.forEach(c => c.dataset.__minPrice = String(extractMinPrice(c)));
      vis.sort((a,b)=>{
        const da = parseFloat(a.dataset.__minPrice), db = parseFloat(b.dataset.__minPrice);
        return sort === 'price_asc' ? (da - db) : (db - da);
      });
      // Re-append visible cards in new order (hidden stay where they are, not an issue visually)
      vis.forEach(c => grid.appendChild(c));
    }

    // update labels (if present)
    const btnSort = root.querySelector('.mlz-dd[data-dd="sort"] .mlz-dd-btn');
    const btnFarm = root.querySelector('.mlz-dd[data-dd="farm"] .mlz-dd-btn');
    if (btnSort){
      btnSort.textContent = sort==='price_asc'  ? 'TRIÉ PAR : PRIX (↓)'
                          : sort==='price_desc' ? 'TRIÉ PAR : PRIX (↑)'
                          : 'TRIÉ PAR : PRIX';
    }
    if (btnFarm){
      btnFarm.textContent = farm ? ('FARMS : ' + farm) : 'FARMS';
    }
  }

  // ---- Derive selection from clicked <a>
  function getChoiceFromLink(a){
    const choice = {sort:null, farm:null};
    if (!a) return choice;
    // prefer data-*
    if (a.hasAttribute('data-sort')) choice.sort = a.getAttribute('data-sort') || null;
    if (a.hasAttribute('data-farm')) choice.farm = a.getAttribute('data-farm') || null;
    // else parse href query (?sort=..., ?farm=...)
    const href = a.getAttribute('href') || '';
    try {
      const u = new URL(href, location.origin);
      const s = u.searchParams.get('sort'); const f = u.searchParams.get('farm');
      if (choice.sort==null && s!=null) choice.sort = s || null;
      if (choice.farm==null && f!=null) choice.farm = f || null;
    } catch(_) {}
    return choice;
  }

  // ---- Delegated clicks (STRICT SCOPE)
  document.addEventListener('click', function(e){
    // open/close menus
    const btn = e.target.closest('#category-list .mlz-dd-btn');
    if (btn){
      const dd  = btn.closest('.mlz-dd');
      const bar = dd && dd.parentElement;
      if (bar){
        bar.querySelectorAll('.mlz-dd.open').forEach(x=>{ if (x!==dd) x.classList.remove('open'); });
      }
      dd && dd.classList.toggle('open');
      return;
    }
    // clicks on items (only menus inside #category-list)
    const link = e.target.closest('#category-list .mlz-dd-menu a');
    if (link){
      const choice = getChoiceFromLink(link);
      if (choice.sort===null && choice.farm===null){
        // not our item -> allow default navigation
        return;
      }
      e.preventDefault(); e.stopPropagation();
      // determine current other choice from active labels if missing
      const root = getRoot();
      if (root){
        if (choice.sort===null){
          const btnSort = root.querySelector('.mlz-dd[data-dd="sort"] .mlz-dd-btn');
          if (btnSort){
            if (btnSort.textContent.includes('(↓)')) choice.sort='price_asc';
            else if (btnSort.textContent.includes('(↑)')) choice.sort='price_desc';
          }
        }
        if (choice.farm===null){
          const active = root.querySelector('.mlz-dd[data-dd="farm"] .mlz-dd-menu a.is-active');
          if (active) choice.farm = active.getAttribute('data-farm') || null;
        }
      }
      applyFilters({ sort: choice.sort || '', farm: choice.farm || '' });
      // close menus
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
      return;
    }
    // click outside -> close
    if (!e.target.closest('#category-list .mlz-dd')){
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
    }
  }, true);

  // ESC to close
  document.addEventListener('keydown', (e)=>{
    if (e.key === 'Escape'){
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
    }
  });

  // Re-apply on injection: if someone set query params already, apply once
  const mo = new MutationObserver(()=>{
    const root = getRoot();
    if (!root) return;
    // Optional auto-apply: read current labels? We skip to avoid any surprise.
  });
  mo.observe(document.body, {childList:true, subtree:true});

  log('mlz-filters-ultra ready');
})();
