/* assets/js/currency.js — ultra-safe, sans écouteurs de clic */
(function () {
  "use strict";

  // Boot depuis PHP
  const BOOT = (window.MLZ_CURRENCY_BOOT || {});
  let CUR = {
    code: BOOT.code || "EUR",
    symbol: BOOT.symbol || "€",
    rate: Number(BOOT.rate || 1),
    dec: Number(BOOT.dec ?? 2),
    pos: BOOT.pos || "suffix",
    decsep: BOOT.decsep || ",",
    thousand: BOOT.thousand || " "
  };

  const $$ = (s, r=document) => Array.from(r.querySelectorAll(s));

  // --- formatage robuste (évite ".00.00")
  function formatMoney(amount) {
    const dec   = Math.max(0, CUR.dec | 0);
    const fixed = amount.toFixed(dec);           // "2725" ou "2725.00"
    let [intPart, decPart] = fixed.split('.');   // on sépare TOUJOURS sur "."
    intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, CUR.thousand);
    const num = dec ? (intPart + CUR.decsep + (decPart ?? "")) : intPart;
    return CUR.pos === "prefix" ? (CUR.symbol + num) : (num + " " + CUR.symbol);
  }

  // --- applique la devise dans un conteneur
  function applyCurrencyWithin(root=document) {
    try {
      $$("[data-price-eur]", root).forEach(node => {
        const base = Number(node.getAttribute("data-price-eur") || 0);
        const converted = base * CUR.rate;
        node.textContent = formatMoney(converted);
        node.setAttribute("data-price-currency", CUR.code);
      });
      $$(".js-currency-symbol", root).forEach(n => n.textContent = CUR.symbol);
    } catch(e){ console.error("[currency] applyCurrencyWithin:", e); }
  }

  // --- helpers UI (label + item actif)
  function labelFor(code){
    return (code==="EUR" ? "Euro (€)" : (code==="USD" ? "Dollar ($)" : "THB (฿)"));
  }
  function updateMenuUI(code){
    try {
      const btn = document.querySelector('.mlz-dd-btn[data-dd="currency"]');
      if (btn) btn.textContent = "DEVISE : " + labelFor(code);
      document.querySelectorAll('.mlz-dd-menu[data-dd="currency"] [data-currency]')
        .forEach(a => a.classList.toggle('is-active', a.getAttribute('data-currency') === code));
    } catch(e){ /* noop */ }
  }

  // --- appel AJAX vers le back (persistance session)
  let SWITCHING = false;
  async function persistCurrency(code) {
    const url = window.MLZ_SET_CURRENCY_URL || "/api/set_currency.php";
    const res = await fetch(url, {
      method: "POST",
      headers: {"Content-Type":"application/x-www-form-urlencoded; charset=UTF-8"},
      credentials: "include",
      body: "code=" + encodeURIComponent(code)
    });
    const data = await res.json();
    if (!data.ok) throw new Error(data.error || "Devise invalide");
    return data;
  }

  // --- API GLOBALE appelée par category-filters.js
  window.mlzSetCurrency = async function(code){
    if (!code || SWITCHING) return false;
    if (code === CUR.code) { updateMenuUI(code); return false; }

    SWITCHING = true;

    // UI immédiate (label + actif) pour ressenti instantané
    updateMenuUI(code);

    try {
      // persist + récup config serveur
      const data = await persistCurrency(code);
      CUR = {
        code: data.code,
        symbol: data.cfg.symbol,
        rate: Number(data.cfg.rate),
        dec: Number(data.cfg.dec),
        pos: data.cfg.pos,
        decsep: data.cfg.decsep,
        thousand: data.cfg.thousand
      };

      // recalcul global des prix
      applyCurrencyWithin(document);

      // event si d’autres widgets écoutent
      document.dispatchEvent(new CustomEvent("mlz:currency-changed", { detail: CUR }));
    } catch (e) {
      console.error("[currency] mlzSetCurrency:", e);
      // on remet l’UI sur la devise actuelle
      updateMenuUI(CUR.code);
    } finally {
      SWITCHING = false;
    }
    return false;
  };

  // --- auto-apply au chargement + sur injections
  document.addEventListener("DOMContentLoaded", () => {
    applyCurrencyWithin(document);
  });

  const mo = new MutationObserver(list => {
    for (const m of list) {
      if (m.type === "childList" && m.addedNodes.length) {
        m.addedNodes.forEach(n => { if (n.nodeType === 1) applyCurrencyWithin(n); });
      }
    }
  });
  mo.observe(document.body, { childList:true, subtree:true });
})();