/* ==========================================================================
   MeltLabz – category.js (DOM-only tri/filtre + nav AJAX déléguée)
   - Promote la farm/micron choisie EN HAUT
   - Matching robuste : chiffres, slash, micro µ/μ -> u, alias
   - Tri prix lit data-price-eur si présent
   ========================================================================== */

(() => {
  if (window.__mlzTriOnce) return;
  window.__mlzTriOnce = true;

  const STATE = { sort: 'default', farm: 'all' }; // sort: default|price-asc|price-desc

  // --- Normalisations & alias ------------------------------------------------
  const MU_RE = /[\u00B5\u03BC]/g;              // µ (micro sign) & μ (grec)
  const NORM = (s) => (s || '')
    .toString()
    .replace(MU_RE, 'u')                        // µ/μ -> u
    .normalize('NFD').replace(/[\u0300-\u036f]/g, '')
    .toLowerCase().replace(/\s+/g, ' ').trim();

  // Canon = on retire tout sauf [a-z0-9]
  const CANON = (s) => NORM(s).replace(/[^a-z0-9]+/g, '');

  // Alias courants
  const ALIASES = new Map([
    // farms
    ['bbf', 'black bear farms'],
    ['blackbearfarms', 'black bear farms'],
    ['piattella', 'piatella'],

    // microns / formats
    ['90120u', '90/120u'],
    ['90120',  '90/120u'],
    ['90u',    '90u'],
    ['120u',   '120u'],
    ['45u',    '45u'],
    ['70u',    '70u'],
    ['73u',    '73u'],
    ['160u',   '160u'],
    ['fullspectrum', 'full spectrum'],
    ['full-spectrum', 'full spectrum'],
    ['hashpower', 'hash power'],
    ['allmicron', 'all'],                       // “All micron” -> all
    ['allmicrons', 'all'],
  ]);

  // --- Sélecteurs / helpers --------------------------------------------------
  const $  = (sel, root = document) => root.querySelector(sel);
  const $$ = (sel, root = document) => Array.from(root.querySelectorAll(sel));

  // prix: privilégie data-price-eur (indépendant de la devise affichée)
  const PRICE_RE = /(\d[\d\s]*(?:[,.]\d+)?)/;
  const COL_RE   = /\bcol(?:$|[-\s])/;
  const isCol = (el) => el && el.nodeType === 1 && COL_RE.test(el.className);

  function priceFromEl(el){
    // <span class="js-price" data-price-eur="300.00">…</span>
    const js = el.querySelector?.('.js-price') || (el.classList?.contains('js-price') ? el : null);
    if (js) {
      const v = parseFloat(js.getAttribute('data-price-eur'));
      if (Number.isFinite(v)) return v;
    }
    const m = String(el.textContent || '').match(PRICE_RE);
    const n = m ? Number(m[1].replace(/\s/g, '').replace(',', '.')) : NaN;
    return Number.isFinite(n) ? n : NaN;
  }

  // Ajoute t au jeu de tags + variantes utiles (90/120u -> 90,120,90u,120u,90120u)
  function pushTagSmart(setRaw, setCanon, t){
    if (!t) return;
    const raw = NORM(t);
    if (!raw) return;

    // base
    setRaw.add(raw);
    setCanon.add(CANON(raw));

    // variante compacte (supprime tout sauf alnum)
    const compact = raw.replace(/[^a-z0-9]+/g, '');
    if (compact && !setCanon.has(compact)) setCanon.add(compact);

    // si présence de chiffres, indexer les morceaux (séparateurs / , - espace)
    if (/\d/.test(raw)){
      const parts = raw.split(/[^a-z0-9]+/i).filter(Boolean);
      parts.forEach(p => {
        const pr = NORM(p);
        if (pr) {
          setRaw.add(pr);
          setCanon.add(CANON(pr));
          // si c’est purement numérique, indexer aussi p+"u" (90 -> 90u)
          if (/^\d+$/.test(pr)) setCanon.add(CANON(pr + 'u'));
        }
      });
    }
  }

  // --- Collecte des métadonnées par carte -----------------------------------
  let META = [];

  const buildMeta = (col) => {
    const card = $('.product-card', col);
    if (!card) return null;

    // Index d'origine (pour “Par défaut”)
    if (!col.dataset.origIndex) {
      const parent = col.parentNode;
      col.dataset.origIndex = String(Array.prototype.indexOf.call(parent.children, col));
    }
    const origIndex = Number(col.dataset.origIndex || 0);

    // Prix minimal (depuis .badge-price)
    let min = Infinity;
    $$('.badge-price', card).forEach(b => {
      const v = priceFromEl(b);
      if (Number.isFinite(v)) min = Math.min(min, v);
    });
    if (!Number.isFinite(min)) min = Infinity;

    // Tags (farm + badges + strain/seller + data-tags)
    const tagsRaw = new Set();
    const tagsCanon = new Set();

    pushTagSmart(tagsRaw, tagsCanon, card.getAttribute('data-farm'));

    const seller = $('.badge-seller', card);
    if (seller) pushTagSmart(tagsRaw, tagsCanon, seller.textContent);

    const strain = $('.badge-strain', card);
    if (strain) pushTagSmart(tagsRaw, tagsCanon, strain.textContent.replace('🌿', ''));

    $$('.badge, [class*="badge"]:not(.badge-price)', card).forEach(b => {
      pushTagSmart(tagsRaw, tagsCanon, b.textContent);
    });

    const extra = card.getAttribute('data-tags');
    if (extra) extra.split(',').forEach(x => pushTagSmart(tagsRaw, tagsCanon, x));

    return { col, card, origIndex, price: min, tagsRaw, tagsCanon };
  };

  const collect = () => {
    const c = $('.tiles') || $('#category-list') || document;
    if (!c) { META = []; return; }
    const cols = $$('[class^="col-"], [class*=" col-"]', c).filter(isCol);
    META = cols.map(buildMeta).filter(Boolean);
  };

  // --- Tri générique ---------------------------------------------------------
  const sortByState = (arr) => {
    if (STATE.sort === 'price-asc')  return arr.slice().sort((a, b) => a.price - b.price);
    if (STATE.sort === 'price-desc') return arr.slice().sort((a, b) => b.price - a.price);
    return arr.slice().sort((a, b) => a.origIndex - b.origIndex);
  };

  // --- Application (promote la sélection en haut) ----------------------------
  const apply = () => {
    const container = $('.tiles') || $('#category-list') || document;
    if (!container) return;

    if (!META.length || !META.every(m => m.col && container.contains(m.col))) collect();

    // valeur recherchée
    let wanted = STATE.farm;
    let wCanon = CANON(wanted);
    if (ALIASES.has(wCanon)) wanted = ALIASES.get(wCanon);
    wCanon = CANON(wanted);

    const numericish = /\d/.test(wCanon);

    const matches = [];
    const others  = [];

    META.forEach(m => {
      m.col.style.display = ''; // on laisse tout visible, on remonte juste les matches
      if (STATE.farm === 'all') {
        matches.push(m);
        return;
      }
      const hasExact = m.tagsCanon.has(wCanon);
      const hasSoft  = numericish && Array.from(m.tagsCanon).some(tc => tc.includes(wCanon));
      (hasExact || hasSoft) ? matches.push(m) : others.push(m);
    });

    const ordered = (STATE.farm === 'all')
      ? sortByState(matches)
      : [...sortByState(matches), ...sortByState(others)];

    const host = $('.tiles .row') || $('.tiles') || container;
    const frag = document.createDocumentFragment();
    ordered.forEach(m => frag.appendChild(m.col));
    host.appendChild(frag);
  };

  // --- Parsing des valeurs de menus -----------------------------------------
  const parseSort = (raw) => {
    const v = NORM(raw).replace(/_/g,'-');
    if (!v || v === 'default' || v.includes('defaut') || v.includes('réinitial')) return 'default';
    if (v.includes('price-asc') || v.includes('asc') || v.includes('moins') || /-\s*cher/.test(v)) return 'price-asc';
    if (v.includes('price-desc') || v.includes('desc') || v.includes('plus')  || /\+\s*cher/.test(v)) return 'price-desc';
    return 'default';
  };
  const parseFarm = (raw) => {
    const v = NORM(raw);
    if (!v || v === 'all' || v.includes('toutes') || v.includes('all micron')) return 'all';
    return v; // alias gérés plus loin
  };

  // --- Délégation: filtres/tri ----------------------------------------------
  document.addEventListener('click', (ev) => {
    // ignorer les clics à l’intérieur d’une card (gérés par nav)
    if (ev.target.closest('.product-card')) return;

    const btn = ev.target.closest('.mlz-filterbar [data-sort], .mlz-filterbar [data-farm]');
    if (!btn) return;

    ev.preventDefault();

    if (btn.hasAttribute('data-sort')) {
      STATE.sort = parseSort(btn.getAttribute('data-sort'));
      apply();
      return;
    }
    if (btn.hasAttribute('data-farm')) {
      STATE.farm = parseFarm(btn.getAttribute('data-farm'));
      apply();
      return;
    }
  }, true);

  // --- Délégation: NAV AJAX sur .js-route / data-href -----------------------
  const routeSelector = '.js-route, .product-card[data-href], a.js-route[href]';

  function tryAjaxNavigate(href){
    const evt = new CustomEvent('mlz:navigate', { detail: { href }, cancelable: true });
    const notCanceled = document.dispatchEvent(evt);
    try {
      if (window.mlzNavigate)                         { window.mlzNavigate(href); return true; }
      if (window.app && typeof app.go === 'function'){ app.go(href);             return true; }
      if (typeof window.routeTo === 'function')      { window.routeTo(href);     return true; }
      if (typeof window.navigateTo === 'function')   { window.navigateTo(href);  return true; }
      if (!notCanceled)                               return true;
    } catch(_) {}
    return false;
  }

  document.addEventListener('click', (ev) => {
    const node = ev.target.closest(routeSelector);
    if (!node) return;
    const href = node.getAttribute('data-href') || node.getAttribute('href');
    if (!href) return;
    ev.preventDefault();
    if (tryAjaxNavigate(href)) return;
    window.location.href = href;
  }, true);

  // Accessibilité clavier
  document.addEventListener('keydown', (ev) => {
    if (ev.key !== 'Enter' && ev.key !== ' ') return;
    const node = ev.target.closest(routeSelector);
    if (!node) return;
    const href = node.getAttribute('data-href') || node.getAttribute('href');
    if (!href) return;
    ev.preventDefault();
    if (tryAjaxNavigate(href)) return;
    window.location.href = href;
  }, true);

  // --- Observer: réapplique après remplacements AJAX de .tiles --------------
  const mo = new MutationObserver((muts) => {
    let touchTiles = false;
    for (const m of muts) {
      if (m.type !== 'childList') continue;
      if (m.target?.classList?.contains('tiles')) { touchTiles = true; break; }
      if (Array.from(m.addedNodes).some(n => n.nodeType === 1 && (n.classList?.contains?.('tiles') || n.querySelector?.('.tiles')))) {
        touchTiles = true; break;
      }
    }
    if (touchTiles) requestAnimationFrame(() => { collect(); apply(); });
  });
  mo.observe(document.body, { childList: true, subtree: true });

  // Premier passage
  document.addEventListener('DOMContentLoaded', () => { collect(); apply(); });
})();

/* ===================== DEVISE (EUR / USD / THB) – 1 clic, AJAX, formatage ===================== */
(() => {
  if (window.__mlzCurrencyOnce) return;
  window.__mlzCurrencyOnce = true;

  const BOOT = window.MLZ_CURRENCY_BOOT || {};
  let CUR = {
    code: (BOOT.code || 'EUR').toUpperCase(),
    symbol: BOOT.symbol || '€',
    rate: Number(BOOT.rate || 1),
    dec: Number(BOOT.dec ?? 2),
    pos: BOOT.pos || 'suffix',
    decsep: BOOT.decsep || ',',
    thousand: BOOT.thousand || ' '
  };

  const $$ = (s, r=document)=>Array.from(r.querySelectorAll(s));

  function labelFor(code){
    code = (code||'EUR').toUpperCase();
    return code==='EUR' ? 'Euro (€)' : code==='USD' ? 'Dollar ($)' : 'THB (฿)';
  }

  function formatMoney(amount){
    const dec   = Math.max(0, CUR.dec | 0);
    const fixed = Number(amount||0).toFixed(dec);
    let [intPart, decPart] = fixed.split('.');
    intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, CUR.thousand);
    const num = dec ? (intPart + CUR.decsep + (decPart ?? '')) : intPart;
    return (CUR.pos === 'prefix') ? (CUR.symbol + num) : (num + ' ' + CUR.symbol);
  }

  function applyCurrencyWithin(root=document){
    $$('.js-price[data-price-eur]', root).forEach(el=>{
      const eur = parseFloat(el.getAttribute('data-price-eur'));
      if (!Number.isFinite(eur)) return;
      el.textContent = formatMoney(eur * CUR.rate);
      el.setAttribute('data-price-currency', CUR.code);
    });
    $$('.js-currency-symbol', root).forEach(n => n.textContent = CUR.symbol);
  }

  function updateMenuUI(code){
    const btn = document.querySelector('.mlz-dd-btn[data-dd="currency"]');
    if (btn) {
      const prefix =
        btn.getAttribute('data-i18n-currency-btn') ||
        (window.I18N && window.I18N['currency.button']) ||
        'DEVISE';
      btn.textContent = prefix + ' : ' + labelFor(code);
    }
    document.querySelectorAll('.mlz-dd-menu[data-dd="currency"] [data-currency]')
      .forEach(a => a.classList.toggle('is-active',
        (a.getAttribute('data-currency')||'').toUpperCase() === code));
  }

  async function persist(code){
    const cand = [
      window.MLZ_SET_CURRENCY_URL || '',
      'currency_set.php',
      '/api/set_currency.php'
    ].filter(Boolean);

    for (const url of cand){
      try{
        const res = await fetch(url, {
          method:'POST',
          headers:{'Content-Type':'application/x-www-form-urlencoded; charset=UTF-8'},
          credentials:'include',
          body:'code='+encodeURIComponent(code)
        });
        const data = await res.json();
        if (data && data.ok) return data;
      }catch(_){}
    }
    const fallback = (c)=>({
      ok:true,
      code:c,
      cfg: (c==='USD') ? {symbol:'$', rate:1.0, dec:0, pos:'prefix', decsep:'.', thousand:','}
          : (c==='THB')? {symbol:'฿', rate:1.0, dec:0, pos:'suffix', decsep:'.', thousand:','}
                        : {symbol:'€', rate:1.0, dec:2, pos:'suffix', decsep:',', thousand:' '}
    });
    return fallback(code);
  }

  let SWITCHING = false;
  async function setCurrency(code){
    code = (code||'EUR').toUpperCase();
    if (SWITCHING || code === CUR.code) { updateMenuUI(code); return; }
    SWITCHING = true;

    updateMenuUI(code); // feedback immédiat

    try{
      const data = await persist(code);
      const cfg  = data.cfg || {};
      CUR = {
        code: (data.code || code).toUpperCase(),
        symbol: cfg.symbol ?? (code==='USD'?'$':code==='THB'?'฿':'€'),
        rate: Number(cfg.rate ?? 1),
        dec: Number(cfg.dec ?? (code==='USD'||code==='THB'?0:2)),
        pos: cfg.pos ?? (code==='EUR'?'suffix':'prefix'),
        decsep: cfg.decsep ?? (code==='EUR'?',':'.'),
        thousand: cfg.thousand ?? (code==='EUR'?' ':',')
      };

      applyCurrencyWithin(document);
      setTimeout(()=>applyCurrencyWithin(document), 0);
      setTimeout(()=>applyCurrencyWithin(document), 150);

      document.dispatchEvent(new CustomEvent('mlz:currency-changed', { detail: CUR }));
    }catch(e){
      console.error('[currency] setCurrency:', e);
      updateMenuUI(CUR.code);
    }finally{
      SWITCHING = false;
    }
  }

  function onPick(ev){
    const a = ev.target.closest('.mlz-dd-menu[data-dd="currency"] [data-currency]');
    if (!a) return;
    ev.preventDefault();
    ev.stopImmediatePropagation();
    const code = (a.getAttribute('data-currency') || 'EUR').toUpperCase();
    setCurrency(code);
    const dd = a.closest('.mlz-dd'); if (dd) dd.classList.remove('open','is-open');
    if (document.activeElement && document.activeElement.blur) document.activeElement.blur();
  }
  document.addEventListener('pointerdown', onPick, true);
  document.addEventListener('click',       onPick, true);

  document.addEventListener('change', (e)=>{
    const sel = e.target && e.target.id === 'currency-select' ? e.target : null;
    if (!sel) return;
    setCurrency(sel.value);
  }, true);

  document.addEventListener('DOMContentLoaded', () => {
    updateMenuUI(CUR.code);
    applyCurrencyWithin(document);
  });

  const moCurrency = new MutationObserver((muts)=>{
    if (muts.some(m =>
      Array.from(m.addedNodes||[]).some(n =>
        n.nodeType===1 && (n.querySelector?.('.mlz-dd-menu[data-dd="currency"]') ||
                           n.matches?.('.mlz-dd-menu[data-dd="currency"]') ||
                           n.querySelector?.('.js-price[data-price-eur]'))
      )
    )){
      updateMenuUI(CUR.code);
      applyCurrencyWithin(document);
    }
  });
  moCurrency.observe(document.body, { childList: true, subtree: true });

  window.mlzSetCurrency = (c)=>setCurrency(String(c||'').toUpperCase());
})();

/* === UI TRI PRIX & FARMS : libellés i18n + is-active ======================== */
(function(){
  let lastSort = null;          
  let lastFarmVal = null;       
  let lastFarmLabel = '';

  const T = (key, fallback) => (window.I18N && window.I18N[key]) || fallback || key;

  const $  = (s, r=document)=>r.querySelector(s);
  const $$ = (s, r=document)=>Array.from(r.querySelectorAll(s));
  const N  = (s)=>String(s||'').replace(/[\u00B5\u03BC]/g,'u').normalize('NFD').replace(/[\u0300-\u036f]/g,'').toLowerCase().trim();

  function cleanLabel(el){ return (el && el.textContent ? el.textContent.replace(/\s*\(\d+\)\s*$/,'') : '').trim(); }
  function parseSort(val){
    const v = N(val).replace(/_/g,'-');
    if (v.includes('price-asc') || /(^|-)asc$/.test(v) || v.includes('moins')) return 'asc';
    if (v.includes('price-desc')|| /(^|-)desc$/.test(v) || v.includes('plus'))  return 'desc';
    return 'default';
  }

  function findSortBtn(){
    return document.querySelector(
      '.mlz-filterbar [data-dd="sort"] .mlz-dd-btn,'+
      '.mlz-filterbar .mlz-dd-btn[data-dd="sort"],'+
      '.mlz-filterbar [data-dd="sort"] button'
    );
  }
  function findFarmBtn(){
    return document.querySelector(
      '.mlz-filterbar [data-dd="farm"] .mlz-dd-btn,'+
      '.mlz-filterbar .mlz-dd-btn[data-dd="farm"],'+
      '.mlz-filterbar [data-dd="farm"] button'
    );
  }

  function currentSort(){
    const a = $('.mlz-filterbar [data-sort].is-active');
    if (a) return parseSort(a.getAttribute('data-sort'));
    if (lastSort) return lastSort;
    return parseSort(new URLSearchParams(location.search).get('sort'));
  }
  function currentFarm(){
    const a = $('.mlz-filterbar [data-farm].is-active');
    if (a){
      const v = (a.getAttribute('data-farm')||'').trim() || 'all';
      return { v, label: cleanLabel(a) || v };
    }
    if (lastFarmVal) return { v:lastFarmVal, label:lastFarmLabel||lastFarmVal };
    const q = (new URLSearchParams(location.search).get('farm')||'').trim();
    return (!q || N(q)==='all') ? { v:'all', label:'' } : { v:q, label:q };
  }

  let curSymbol = (window.MLZ_CURRENCY_BOOT && MLZ_CURRENCY_BOOT.symbol) || '€';
  document.addEventListener('mlz:currency-changed', (e) => {
    if (e && e.detail && e.detail.symbol) curSymbol = e.detail.symbol;
    requestAnimationFrame(updateLabels);
  });

  function updateMenuI18N(){
    document.querySelectorAll('.mlz-filterbar [data-dd="sort"] [data-sort]').forEach(a=>{
      const raw = (a.getAttribute('data-sort')||'').toLowerCase().replace(/_/g,'-');
      let key, fallback;
      if (raw.includes('price-asc') || /(^|-)asc$/.test(raw)){
        key = "filter.sort.menu.asc";     fallback = "Du – cher au + cher";
      } else if (raw.includes('price-desc') || /(^|-)desc$/.test(raw)){
        key = "filter.sort.menu.desc";    fallback = "Du + cher au – cher";
      } else {
        key = "filter.sort.menu.default"; fallback = "Par défaut";
      }
      a.textContent = T(key, fallback);
    });

    document.querySelectorAll('.mlz-filterbar [data-dd="farm"] [data-farm]').forEach(a=>{
      const val = (a.getAttribute('data-farm')||'').trim();
      if (!val || N(val) === 'all'){
        const lbl   = T("filter.farm.menu.all", "Toutes les farms");
        const count = a.querySelector('.count');
        a.textContent = lbl;
        if (count) a.appendChild(count);
      }
    });
  }

  function updateLabels(){
    const sortBtn = findSortBtn();
    if (sortBtn){
      const s = currentSort();
      if (s === 'asc'){
       sortBtn.textContent = T("filter.sort.button.asc", "TRIÉ PAR : PRIX ") + curSymbol + "↓";
      } else if (s === 'desc'){
       sortBtn.textContent = T("filter.sort.button.desc", "TRIÉ PAR : PRIX ") + curSymbol + "↑";
      } else {
        sortBtn.textContent = T("filter.sort.button.base", "TRIÉ PAR : PRIX");
      }
    }
    const farmBtn = findFarmBtn();
    if (farmBtn){
      const f = currentFarm();
      farmBtn.textContent = (f.v && N(f.v) !== 'all')
        ? T("filter.farm.button.sel", "FARMS : %s").replace("%s", f.label)
        : T("filter.farm.button.base", "FARMS");
    }
    updateMenuI18N();
  }

  function setActiveInMenu(el){
    const menu = el.closest('.mlz-dd-menu') || el.parentElement;
    if (!menu) return;
    $$('.is-active', menu).forEach(n=>n.classList.remove('is-active'));
    el.classList.add('is-active');
  }

  document.addEventListener('DOMContentLoaded', updateLabels);

  document.addEventListener('click', function(e){
    const el = e.target.closest && e.target.closest('.mlz-filterbar [data-sort], .mlz-filterbar [data-farm]');
    if (!el) return;

    if (el.hasAttribute('data-sort')){
      lastSort = parseSort(el.getAttribute('data-sort'));
      setActiveInMenu(el);
    } else if (el.hasAttribute('data-farm')){
      lastFarmVal   = (el.getAttribute('data-farm')||'').trim() || 'all';
      lastFarmLabel = cleanLabel(el) || lastFarmVal;
      setActiveInMenu(el);
    }
    requestAnimationFrame(updateLabels);
  }, false);

  const mo = new MutationObserver((muts)=>{
    if (muts.some(m => Array.from(m.addedNodes||[]).some(n => n.nodeType===1 && (n.matches?.('.mlz-filterbar') || n.querySelector?.('.mlz-filterbar'))))) {
      requestAnimationFrame(updateLabels);
    }
  });
  mo.observe(document.body, { childList:true, subtree:true });

  document.addEventListener('mlz:tiles-ready', updateLabels);
})();