/* assets/js/category-filters.js — tri, farms **et devise** (AJAX, no reload, 1 seul clic) */
(function(){
  const doc = document;
  const norm = s => (s||'').toString().toLowerCase().trim();

  /* ----------------------------- CURRENCY (merged) ----------------------------- */

  // Boot côté PHP (require: window.MLZ_CURRENCY_BOOT défini dans la page)
  const BOOT = (window.MLZ_CURRENCY_BOOT || {});
  let CUR = {
    code: BOOT.code || "EUR",
    symbol: BOOT.symbol || "€",
    rate: Number(BOOT.rate || 1),
    dec: Number(BOOT.dec ?? 2),
    pos: BOOT.pos || "suffix",
    decsep: BOOT.decsep || ",",
    thousand: BOOT.thousand || " "
  };

  function formatMoney(amount) {
    const dec   = Math.max(0, CUR.dec | 0);
    const fixed = amount.toFixed(dec);           // "2725" ou "2725.00"
    let [intPart, decPart] = fixed.split('.');   // sépare TOUJOURS sur "."
    intPart = intPart.replace(/\B(?=(\d{3})+(?!\d))/g, CUR.thousand);
    const num = dec ? (intPart + CUR.decsep + (decPart ?? "")) : intPart;
    return CUR.pos === "prefix" ? (CUR.symbol + num) : (num + " " + CUR.symbol);
  }

  function applyCurrencyWithin(root=document) {
    try {
      Array.from(root.querySelectorAll("[data-price-eur]")).forEach(node => {
        const base = Number(node.getAttribute("data-price-eur") || 0);
        const converted = base * CUR.rate;
        node.textContent = formatMoney(converted);
        node.setAttribute("data-price-currency", CUR.code);
      });
      Array.from(root.querySelectorAll(".js-currency-symbol")).forEach(n => n.textContent = CUR.symbol);
    } catch(e){ console.error("[currency] applyCurrencyWithin:", e); }
  }

  function labelFor(code){
    return (code==="EUR" ? "Euro (€)" : (code==="USD" ? "Dollar ($)" : "THB (฿)"));
  }
  function updateCurrencyUI(code){
    try {
      const btn = document.querySelector('.mlz-dd-btn[data-dd="currency"]');
      if (btn) btn.textContent = "DEVISE : " + labelFor(code);
      document.querySelectorAll('.mlz-dd-menu[data-dd="currency"] [data-currency]')
        .forEach(a => a.classList.toggle('is-active', a.getAttribute('data-currency') === code));
    } catch(e){ /* noop */ }
  }

  let CUR_SWITCHING = false;
  async function setCurrency(code){
    if (!code || code === CUR.code || CUR_SWITCHING) return;
    CUR_SWITCHING = true;
    const url = window.MLZ_SET_CURRENCY_URL || "/api/set_currency.php";
    try {
      // feedback immédiat (label/actif) pour 1er clic ressenti instantané
      updateCurrencyUI(code);

      const res = await fetch(url, {
        method: "POST",
        headers: {"Content-Type":"application/x-www-form-urlencoded; charset=UTF-8"},
        credentials: "include",
        body: "code=" + encodeURIComponent(code)
      });
      const data = await res.json();
      if (!data.ok) throw new Error(data.error || "Devise invalide");

      CUR = {
        code: data.code,
        symbol: data.cfg.symbol,
        rate: Number(data.cfg.rate),
        dec: Number(data.cfg.dec),
        pos: data.cfg.pos,
        decsep: data.cfg.decsep,
        thousand: data.cfg.thousand
      };

      applyCurrencyWithin(document);
      // “sécurité” si d’autres scripts réécrivent le DOM juste après :
      setTimeout(()=>applyCurrencyWithin(document), 0);
      setTimeout(()=>applyCurrencyWithin(document), 180);

      document.dispatchEvent(new CustomEvent("mlz:currency-changed", { detail: CUR }));
    } catch (e) {
      console.error("[currency] setCurrency error:", e);
      // revert UI sur la devise effective
      updateCurrencyUI(CUR.code);
    } finally {
      CUR_SWITCHING = false;
    }
  }

  /* ------------------------------- SORT / FARMS ------------------------------- */

  function getParam(n){
    const p = new URLSearchParams(location.search);
    return p.get(n) || '';
  }
  function setParamsInUrl(params){
    const p = new URLSearchParams(location.search);
    if ('sort' in params){ params.sort ? p.set('sort', params.sort) : p.delete('sort'); }
    if ('farm' in params){ params.farm ? p.set('farm', params.farm) : p.delete('farm'); }
    const url = location.pathname + '?' + p.toString();
    try { history.pushState({view:'catlist'}, '', url); } catch(_){}
  }

  function getCat(){
    const box = doc.getElementById('category-list');
    const c1  = box?.dataset?.categorie;
    return (c1 || getParam('categorie') || 'all').toLowerCase();
  }
  function getListRoot(){
    const root = doc.getElementById('category-list'); if (!root) return null;
    return root.querySelector('.grid-products, .mlz-products, .products-grid, .mlz-wrap, .row, .cards, .cards-wrap') || root;
  }
  function getCards(root){
    return Array.from(root.querySelectorAll('.product-card, a.product-card, .card.product, .card-prod'));
  }

  function minPrice(card){
    // NB: utilisé uniquement pour trier en EUR si le texte contient “€” (fallback)
    const txt = (card.textContent || '').replace(/\s+/g,' ');
    const re  = /([0-9]{1,3}(?:[ .][0-9]{3})*|[0-9]+)(?:,([0-9]{1,2}))?\s*€/g;
    let m, best = Infinity;
    while ((m = re.exec(txt))){
      const n = parseFloat((m[1]||'0').replace(/[ .]/g,'') + '.' + (m[2]||'00'));
      if (!isNaN(n)) best = Math.min(best, n);
    }
    return isFinite(best) ? best : 9e9;
  }
  function farmOf(card){
    const ds = norm(card.getAttribute('data-farm'));
    if (ds) return ds;
    const b = card.querySelector('.farm-badge, .badge.farm, .badge');
    return b ? norm(b.textContent) : '';
  }

  function applySortAndFarm(sort, farm){
    const root = getListRoot(); if (!root) return;
    const cards = getCards(root);
    const want = norm(farm);

    cards.forEach(c => {
      const cf = farmOf(c);
      c.style.display = want ? (cf === want ? '' : 'none') : '';
    });

    if (sort === 'price_asc' || sort === 'price_desc'){
      const vis = cards.filter(c => c.style.display !== 'none');
      vis.forEach(c => c.dataset.__minPrice = String(minPrice(c)));
      vis.sort((a,b) => {
        const da = parseFloat(a.dataset.__minPrice), db = parseFloat(b.dataset.__minPrice);
        return sort === 'price_asc' ? (da - db) : (db - da);
      });
      vis.forEach(c => root.appendChild(c));
    }

    const bar = doc.querySelector('#category-list .mlz-filterbar');
    if (bar){
      const bSort = bar.querySelector('.mlz-dd[data-dd="sort"] .mlz-dd-btn');
      const bFarm = bar.querySelector('.mlz-dd[data-dd="farm"] .mlz-dd-btn');
      if (bSort){
        bSort.textContent = sort==='price_asc'  ? 'TRIÉ PAR : PRIX (↓)'
                            : sort==='price_desc'? 'TRIÉ PAR : PRIX (↑)'
                            : 'TRIÉ PAR : PRIX';
      }
      if (bFarm){
        bFarm.textContent = farm ? ('FARMS : ' + farm) : 'FARMS';
      }
    }
  }

  function ajaxLoad(params){
    const cat = getCat() || 'all';
    const usp = new URLSearchParams({ categorie: cat });
    const cur = new URLSearchParams(location.search);
    const sort = params.sort ?? cur.get('sort') ?? '';
    const farm = params.farm ?? cur.get('farm') ?? '';
    if (sort) usp.set('sort', sort);
    if (farm) usp.set('farm', farm);

    const url = '/category_list.php?' + usp.toString();
    return fetch(url, { headers:{ 'X-Requested-With':'XMLHttpRequest' }, credentials:'include' })
      .then(r => r.text())
      .then(html => {
        const tiles = document.querySelector('.tiles');
        if (tiles) tiles.innerHTML = html;
        try { history.pushState({view:'catlist', cat}, '', url); } catch(_){}
        // Réappliquer tous les états après injection
        applySortAndFarm(sort, farm);
        applyCurrencyWithin(document); // <=== IMPORTANT
      });
  }

  /* --------------------------- Dropdowns (open/close) -------------------------- */

  doc.addEventListener('click', function(e){
    const btn = e.target.closest('#category-list .mlz-dd-btn');
    if (btn){
      const dd  = btn.closest('.mlz-dd');
      const bar = dd?.parentElement;
      bar?.querySelectorAll('.mlz-dd.open').forEach(x=>{ if (x!==dd) x.classList.remove('open'); });
      dd?.classList.toggle('open');
      return;
    }
    if (!e.target.closest('#category-list .mlz-dd')){
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=> x.classList.remove('open'));
    }
  }, true);
  doc.addEventListener('keydown', (e)=>{ if (e.key==='Escape') document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open')); });

  /* ---------------------------- Clicks on items ---------------------------- */
  // Phase capture (comme le reste), pour 1 seul clic.

  // 1) Currency
  const currencyPick = function(e){
    const c = e.target.closest('#category-list .mlz-dd-menu [data-currency]');
    if (!c) return;
    e.preventDefault(); e.stopPropagation();
    const code = c.getAttribute('data-currency') || '';
    if (code) {
      setCurrency(code); // UI immédiat + AJAX + recalcul prix
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
    }
  };
  doc.addEventListener('pointerdown', currencyPick, true); // devance tout le monde
  doc.addEventListener('click',       currencyPick, true);

  // 2) Sort / 3) Farm (inchangés)
  doc.addEventListener('click', function(e){
    const s = e.target.closest('#category-list .mlz-dd-menu a[data-sort]');
    if (s){
      e.preventDefault(); e.stopPropagation();
      const sort = s.dataset.sort || '';
      const farm = getParam('farm') || '';
      setParamsInUrl({ sort, farm });
      ajaxLoad({ sort, farm }).catch(()=>{
        const url = '/category_list.php?categorie='+encodeURIComponent(getCat())
                  + (sort ? '&sort='+encodeURIComponent(sort):'')
                  + (farm ? '&farm='+encodeURIComponent(farm):'');
        location.assign(url);
      });
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
      return;
    }
    const f = e.target.closest('#category-list .mlz-dd-menu a[data-farm]');
    if (f){
      e.preventDefault(); e.stopPropagation();
      const farm = f.dataset.farm || '';
      const sort = getParam('sort') || '';
      setParamsInUrl({ sort, farm });
      ajaxLoad({ sort, farm }).catch(()=>{
        const url = '/category_list.php?categorie='+encodeURIComponent(getCat())
                  + (sort ? '&sort='+encodeURIComponent(sort):'')
                  + (farm ? '&farm='+encodeURIComponent(farm):'');
        location.assign(url);
      });
      document.querySelectorAll('#category-list .mlz-dd.open').forEach(x=>x.classList.remove('open'));
      return;
    }
  }, true);

  /* ------------- Garder sort/farm en naviguant depuis les tuiles ------------- */
  doc.addEventListener('click', function(e){
    const a = e.target.closest('.tiles a[href*="category_list.php"]');
    if (!a) return;
    const url = new URL(a.getAttribute('href'), location.origin);
    const sort = getParam('sort'); const farm = getParam('farm');
    if (sort) url.searchParams.set('sort', sort);
    if (farm) url.searchParams.set('farm', farm);
    e.preventDefault(); e.stopPropagation();
    fetch(url.toString(), { headers:{ 'X-Requested-With':'XMLHttpRequest' }, credentials:'include' })
      .then(r=>r.text())
      .then(html=>{
        const tiles = document.querySelector('.tiles'); if (tiles) tiles.innerHTML = html;
        try { history.pushState({view:'catlist', cat:url.searchParams.get('categorie')||'all'}, '', url.toString()); } catch(_){}
        applySortAndFarm(sort || '', farm || '');
        applyCurrencyWithin(document); // <=== IMPORTANT
      })
      .catch(()=> location.assign(url.toString()));
  }, true);

  /* ---------------- Re-apply on injection / dynamic content ---------------- */
  const mo = new MutationObserver(()=>{
    if (doc.getElementById('category-list')){
      const sort = getParam('sort') || '';
      const farm = getParam('farm') || '';
      if (sort || farm) applySortAndFarm(sort, farm);
      applyCurrencyWithin(document); // <=== IMPORTANT
    }
  });
  mo.observe(doc.body, {childList:true, subtree:true});

  /* -------------------------- Init on first load -------------------------- */
  document.addEventListener('DOMContentLoaded', () => {
    applyCurrencyWithin(document);
  });

  // API publique au besoin (pour debug/console)
  window.mlzSetCurrency = (code) => setCurrency(code);

})();