<?php
// apply_promo.php — Vérification + calcul code promo pour le panier courant
if (session_status() === PHP_SESSION_NONE) { session_start(); }

header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/includes/db.php';

$telegram_id = $_SESSION['telegram_id'] ?? null;
if (!$telegram_id) {
  echo json_encode([
    'ok'      => false,
    'message' => 'Vous devez être connecté via Telegram.'
  ]);
  exit;
}

// Si on veut juste vider le code promo
if (!empty($_POST['clear'])) {
  unset(
    $_SESSION['promo_code'],
    $_SESSION['promo_id'],
    $_SESSION['promo_discount'],
    $_SESSION['promo_total_after']
  );

  echo json_encode([
    'ok'           => true,
    'message'      => 'Code promo retiré.',
    'code'         => null,
    'discount_value' => 0,
    'total_before' => null,
    'total_after'  => null,
  ]);
  exit;
}

$code = strtoupper(trim($_POST['code'] ?? ''));
if ($code === '') {
  echo json_encode([
    'ok'      => false,
    'message' => 'Veuillez entrer un code promo.'
  ]);
  exit;
}

try {
  // 1) Récupérer le code promo
  $st = $pdo->prepare("SELECT * FROM promo_codes WHERE code = ? AND is_active = 1");
  $st->execute([$code]);
  $promo = $st->fetch(PDO::FETCH_ASSOC);

  if (!$promo) {
    echo json_encode([
      'ok'      => false,
      'message' => 'Ce code promo est invalide ou désactivé.'
    ]);
    exit;
  }

  $promoId     = (int)$promo['id'];
  $appliesAll  = (int)$promo['applies_to_all'] === 1;
  $minTotal    = $promo['min_order_total'];
  $dtype       = $promo['discount_type'];   // 'percent' ou 'amount'
  $dval        = (float)$promo['discount_value'];

  if ($dval <= 0) {
    echo json_encode([
      'ok'      => false,
      'message' => 'Ce code promo n\'est pas configuré correctement.'
    ]);
    exit;
  }

  // 2) Récupérer le panier courant
  $qItems = $pdo->prepare("
    SELECT
      pa.produit_id,
      p.nom,
      pa.quantite,
      pp.prix
    FROM paniers pa
    JOIN produits p      ON pa.produit_id = p.id
    JOIN produit_prix pp ON pp.produit_id = p.id AND pp.quantite = pa.quantite
    WHERE pa.telegram_id = ?
  ");
  $qItems->execute([$telegram_id]);
  $items = $qItems->fetchAll(PDO::FETCH_ASSOC);

  if (!$items) {
    echo json_encode([
      'ok'      => false,
      'message' => 'Votre panier est vide.'
    ]);
    exit;
  }

  $cartTotal = 0.0;
  foreach ($items as $it) {
    $cartTotal += (float)$it['prix'];
  }

  // 3) Vérifier le minimum de commande
  if ($minTotal !== null && $minTotal !== '' && $cartTotal < (float)$minTotal) {
    echo json_encode([
      'ok'      => false,
      'message' => 'Montant minimum pour ce code : ' . number_format((float)$minTotal, 2, ',', ' ') . ' €.',
      'total_before' => $cartTotal
    ]);
    exit;
  }

  // 4) Déterminer les produits éligibles
  $eligibleTotal = 0.0;
  $eligibleNames = [];

  $allowedIds = [];
  if (!$appliesAll) {
    $stIds = $pdo->prepare("SELECT produit_id FROM promo_code_produits WHERE promo_id = ?");
    $stIds->execute([$promoId]);
    $allowedIds = array_map('intval', array_column($stIds->fetchAll(PDO::FETCH_ASSOC), 'produit_id'));
  }

  foreach ($items as $it) {
    $price = (float)$it['prix'];
    $pid   = (int)$it['produit_id'];

    if ($appliesAll || in_array($pid, $allowedIds, true)) {
      $eligibleTotal += $price;
      $eligibleNames[] = $it['nom'];
    }
  }

  if ($eligibleTotal <= 0) {
    echo json_encode([
      'ok'      => false,
      'message' => 'Ce code ne s\'applique à aucun produit de votre panier.',
      'total_before' => $cartTotal
    ]);
    exit;
  }

  // 5) Calcul de la remise
  if ($dtype === 'percent') {
    $discount = round($eligibleTotal * $dval / 100, 2);
    $discountLabel = '-' . rtrim(rtrim((string)$dval, '0'), '.') . '%';
  } else {
    $discount = min($dval, $eligibleTotal);
    $discountLabel = '-' . number_format($discount, 2, ',', ' ') . ' €';
  }

  if ($discount > $cartTotal) {
    $discount = $cartTotal;
  }

  $finalTotal = max(0, $cartTotal - $discount);

  // 6) Sauvegarde en session pour le checkout
  $_SESSION['promo_code']        = $code;
  $_SESSION['promo_id']          = $promoId;
  $_SESSION['promo_discount']    = $discount;
  $_SESSION['promo_total_after'] = $finalTotal;

  echo json_encode([
    'ok'             => true,
    'message'        => 'Code ' . $code . ' appliqué.',
    'code'           => $code,
    'discount_type'  => $dtype,
    'discount_value' => $discount,
    'discount_label' => $discountLabel,
    'total_before'   => $cartTotal,
    'total_after'    => $finalTotal,
    'eligible_names' => $eligibleNames,
  ]);
} catch (Throwable $e) {
  echo json_encode([
    'ok'      => false,
    'message' => 'Erreur serveurs, réessaie dans un instant.'
  ]);
}