<?php
// api/scan_once.php — Watcher mempool.space + crédits wallet (VERBOSE)
// CLI:  php api/scan_once.php [--only-address=bc1...] [--nolog]
// HTTP: GET /api/scan_once.php?token=API_TOKEN
declare(strict_types=1);

require __DIR__.'/config.php'; // pdo(), API_TOKEN, etc.

$LOG_FILE = '/home/meltxpwi/logs/btc_scan.log';
$NOLOG    = in_array('--nolog', $argv ?? [], true);

// ---------- Auth (HTTP only) ----------
if (php_sapi_name() !== 'cli') {
  $token = $_GET['token'] ?? $_SERVER['HTTP_AUTHORIZATION'] ?? '';
  if (str_starts_with($token, 'Bearer ')) $token = substr($token, 7);
  if (!defined('API_TOKEN') || $token !== API_TOKEN) {
    http_response_code(401);
    echo json_encode(['ok'=>false,'error'=>'unauthorized']); exit;
  }
}

// ---------- Logging helpers ----------
function logv(string $msg): void {
  global $LOG_FILE, $NOLOG;
  $line = '['.gmdate('Y-m-d H:i:s').' UTC] '.$msg."\n";
  if (!$NOLOG) @file_put_contents($LOG_FILE, $line, FILE_APPEND);
}

// ---------- HTTP helpers (cURL + fallback) ----------
function http_get(string $url): string|false {
  // Try cURL first
  if (function_exists('curl_init')) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_CONNECTTIMEOUT => 10,
      CURLOPT_TIMEOUT        => 20,
      CURLOPT_SSL_VERIFYPEER => true,
      CURLOPT_USERAGENT      => 'MeltLabzScanner/1.0'
    ]);
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);
    if ($resp !== false && $code >= 200 && $code < 300) return $resp;
    logv("cURL fail url=$url code=$code err=$err");
  }
  // Fallback file_get_contents
  $ctx = stream_context_create(['http'=>[
    'timeout'=>20,
    'header'=>"User-Agent: MeltLabzScanner/1.0\r\n"
  ]]);
  $raw = @file_get_contents($url, false, $ctx);
  if ($raw === false) logv("fgc fail url=$url");
  return $raw;
}
function http_json(string $url): array {
  $raw = http_get($url);
  if ($raw === false) return [];
  $j = json_decode($raw, true);
  if (!is_array($j)) { logv("json decode fail url=$url"); return []; }
  return $j;
}

// ---------- Tip height (best effort) ----------
function tip_height(): int {
  $raw = http_get('https://mempool.space/api/blocks/tip/height');
  return ($raw !== false && is_numeric(trim($raw))) ? (int)$raw : 0;
}

// ---------- EUR/BTC rate (self-contained to avoid undefined function) ----------
function btc_eur_rate(): float {
  static $cached = null;
  if ($cached !== null) return $cached;
  $j = http_json('https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=eur');
  $rate = (float)($j['bitcoin']['eur'] ?? 0);
  $cached = $rate;
  return $rate;
}

// ---------- Parse CLI args ----------
$only = null;
if (php_sapi_name() === 'cli') {
  foreach (($argv ?? []) as $a) {
    if (str_starts_with($a, '--only-address=')) $only = substr($a, 15);
  }
}

// ---------- Main ----------
$start = microtime(true);
logv("===== SCAN START — only=".($only ?: 'ALL')." =====");

$pdo = pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

// Addresses to scan
$sql = "SELECT telegram_id, btc_address FROM telegram_users WHERE btc_address IS NOT NULL AND btc_address <> ''";
if ($only) $sql .= " AND btc_address = :addr";
$st = $pdo->prepare($sql);
if ($only) $st->bindValue(':addr', $only, PDO::PARAM_STR);
$st->execute();
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

$height = tip_height();
$rate   = btc_eur_rate();

logv("TipHeight=$height EUR/BTC=$rate rows=".count($rows));

$report = ['scanned'=>0,'new_deposits'=>0,'credited'=>0,'updated'=>0,'errors'=>[]];

foreach ($rows as $row) {
  $tid  = (int)$row['telegram_id'];
  $addr = trim((string)$row['btc_address']);
  if ($addr === '') continue;

  $report['scanned']++;
  logv("Scan addr=$addr (tid=$tid)");

  $txs = http_json("https://mempool.space/api/address/{$addr}/txs");
  if (!$txs) { $report['errors'][] = "no_txs:$addr"; logv("no_txs"); continue; }

  foreach ($txs as $tx) {
    $txid = (string)($tx['txid'] ?? '');

    // Sum outputs to OUR address (handles multiple vouts to same addr)
    $sats = 0;
    if (isset($tx['vout'])) {
      foreach ($tx['vout'] as $o) {
        if (($o['scriptpubkey_address'] ?? '') === $addr) $sats += (int)($o['value'] ?? 0);
      }
    } elseif (isset($tx['outputs'])) {
      foreach ($tx['outputs'] as $o) {
        if (($o['address'] ?? '') === $addr) $sats += (int)round((float)($o['value'] ?? 0) * 1e8);
      }
    }
    if ($sats <= 0) continue;

    // Confirmations
    $confirmed = (bool)($tx['status']['confirmed'] ?? false);
    $blkH      = (int)($tx['status']['block_height'] ?? 0);
    $confs     = ($confirmed && $height > 0 && $blkH > 0) ? max(1, $height - $blkH + 1) : 0;
    $status    = $confs >= 1 ? 'confirmed' : 'pending';

    logv("txid=$txid sats=$sats confs=$confs status=$status");

    // INSERT/UPDATE btc_deposits (unique txid+address d’après ton dump)
    $pdo->beginTransaction();
    try {
      $pdo->prepare(
        "INSERT INTO btc_deposits (telegram_id,address,txid,amount_sats,confirmations,status)
         VALUES (?,?,?,?,?,?)"
      )->execute([$tid,$addr,$txid,$sats,$confs,$status]);
      $report['new_deposits']++;
      $pdo->commit();
      logv("inserted deposit");
    } catch (PDOException $e) {
      if ((int)$e->getCode() === 23000) { // duplicate
        $pdo->rollBack();
        $pdo->prepare(
          "UPDATE btc_deposits SET confirmations=?, status=? WHERE txid=? AND address=?"
        )->execute([$confs,$status,$txid,$addr]);
        $report['updated']++;
        logv("updated deposit");
      } else {
        $pdo->rollBack();
        $msg = "db:".$e->getMessage();
        $report['errors'][] = $msg;
        logv($msg);
        continue;
      }
    }

    // Credit if confirmed (idempotent via wallet_tx.ref = txid)
    if ($confs >= 1) {
      $chk = $pdo->prepare("SELECT id FROM wallet_tx WHERE telegram_id=? AND ref=? AND direction='credit'");
      $chk->execute([$tid,$txid]);
      if (!$chk->fetch()) {
        $eur_per_btc = $rate > 0 ? $rate : 0.0;
        $eur_total   = $eur_per_btc > 0 ? ($sats/1e8) * $eur_per_btc : 0.0;

        $pdo->beginTransaction();
        try {
          $pdo->prepare(
            "INSERT INTO wallet_tx (telegram_id,direction,amount_sats,eur_total,eur_per_btc,ref)
             VALUES (?,?,?,?,?,?)"
          )->execute([$tid,'credit',$sats,$eur_total,$eur_per_btc,$txid]);

          $pdo->prepare("UPDATE telegram_users SET balance_sats = balance_sats + ? WHERE telegram_id=?")
              ->execute([$sats,$tid]);

          $pdo->commit();
          $report['credited']++;
          logv("credited +$sats sats");
        } catch (Throwable $e) {
          $pdo->rollBack();
          $msg = "credit:".$e->getMessage();
          $report['errors'][] = $msg;
          logv($msg);
        }
      } else {
        logv("already credited (wallet_tx)");
      }
    }
  }
}

$dur = round((microtime(true)-$start)*1000);
logv("===== SCAN END in {$dur}ms — report: ".json_encode($report)." =====");

$out = ['ok'=>true] + $report;
if (php_sapi_name()==='cli') {
  echo json_encode($out, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES).PHP_EOL;
} else {
  header('Content-Type: application/json');
  echo json_encode($out);
}
