<?php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

require __DIR__ . '/config.php';   // fournit pdo(), require_auth_melt(), etc.

// ====== CONFIG ======
const CONFIRM_TARGET = 1;  // nombre de confirmations requises avant crédit

// ====== INPUT ======
$in  = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$tid = (int)($in['telegram_id'] ?? 0);
$txid = trim((string)($in['txid'] ?? ''));        // optionnel si crédit manuel
$vout = isset($in['vout']) ? (int)$in['vout'] : null; // idem
$sats = (int)($in['sats'] ?? 0);
$confs = (int)($in['confs'] ?? 0);
$address = trim((string)($in['address'] ?? ''));

if (!$tid || $sats <= 0) {
  echo json_encode(['ok'=>false, 'error'=>'bad_request', 'detail'=>'telegram_id et sats requis']); exit;
}

// Auth (tu peux alléger si tu appelles côté serveur uniquement)
if (function_exists('require_auth_melt')) {
  require_auth_melt($tid);
} elseif (function_exists('require_auth')) {
  require_auth();
}

// ====== DB SCHEMA (robuste) ======
function ensure_schema(PDO $pdo): void {
  // historique des dépôts
  $pdo->exec("
    CREATE TABLE IF NOT EXISTS btc_deposits (
      id INT AUTO_INCREMENT PRIMARY KEY,
      telegram_id BIGINT NOT NULL,
      address VARCHAR(128) NULL,
      txid VARCHAR(128) NULL,
      vout INT NULL,
      sats BIGINT NOT NULL,
      confs INT NOT NULL DEFAULT 0,
      status ENUM('pending','confirmed','manual') NOT NULL DEFAULT 'pending',
      credited TINYINT(1) NOT NULL DEFAULT 0,
      created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
      updated_at TIMESTAMP NULL DEFAULT NULL,
      UNIQUE KEY uk_tx (txid, vout),
      KEY (telegram_id),
      KEY (status),
      KEY (credited)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");

  // table utilisateurs si jamais absente
  $pdo->exec("
    CREATE TABLE IF NOT EXISTS telegram_users (
      telegram_id BIGINT NOT NULL UNIQUE,
      balance_sats BIGINT NOT NULL DEFAULT 0,
      btc_index INT NOT NULL DEFAULT 0,
      btc_address VARCHAR(128) NULL,
      created_at TIMESTAMP NULL,
      last_seen TIMESTAMP NULL,
      is_banned TINYINT(1) NOT NULL DEFAULT 0
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
  ");
}

try {
  $pdo = pdo();
  ensure_schema($pdo);

  $pdo->beginTransaction();

  // verrouille la ligne user
  $lock = $pdo->prepare("SELECT balance_sats FROM telegram_users WHERE telegram_id=? FOR UPDATE");
  $lock->execute([$tid]);
  if (!$lock->fetch()) {
    // crée la ligne user s'il n'existe pas
    $pdo->prepare("INSERT IGNORE INTO telegram_users(telegram_id, balance_sats) VALUES(?, 0)")
        ->execute([$tid]);
  }

  $now = date('Y-m-d H:i:s');

  // --- Cas CRÉDIT MANUEL (pas de txid) -------------------------------------
  if ($txid === '' || $vout === null) {
    $manualId = 'manual:'.bin2hex(random_bytes(6));

    // enregistre la ligne "manual" (pas d’unicité txid/vout)
    $ins = $pdo->prepare("
      INSERT INTO btc_deposits (telegram_id, address, txid, vout, sats, confs, status, credited, created_at, updated_at)
      VALUES (?, ?, ?, NULL, ?, ?, 'manual', 1, ?, ?)
    ");
    $ins->execute([$tid, $address ?: null, $manualId, $sats, max($confs, CONFIRM_TARGET), $now, $now]);

    // crédit atomique
    $pdo->prepare("UPDATE telegram_users SET balance_sats = balance_sats + ? WHERE telegram_id=?")
        ->execute([$sats, $tid]);

    $pdo->commit();
    echo json_encode(['ok'=>true, 'mode'=>'manual', 'credited'=>true, 'delta_sats'=>$sats, 'txid'=>$manualId]);
    exit;
  }

  // --- Cas ON-CHAIN (txid/vout fournis) ------------------------------------
  // essaie de récupérer la tx (verrou implicite via SELECT ... FOR UPDATE si tu veux renforcer)
  $sel = $pdo->prepare("SELECT id, credited, confs, status FROM btc_deposits WHERE txid=? AND vout=? LIMIT 1");
  $sel->execute([$txid, $vout]);
  $row = $sel->fetch(PDO::FETCH_ASSOC);

  if (!$row) {
    // nouvelle entrée
    $status = ($confs >= CONFIRM_TARGET) ? 'confirmed' : 'pending';
    $credited = ($confs >= CONFIRM_TARGET) ? 1 : 0;

    $ins = $pdo->prepare("
      INSERT INTO btc_deposits
        (telegram_id, address, txid, vout, sats, confs, status, credited, created_at, updated_at)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $ins->execute([$tid, $address ?: null, $txid, $vout, $sats, $confs, $status, $credited, $now, $now]);

    if ($credited) {
      // premier enregistrement déjà confirmé → crédite immédiatement
      $pdo->prepare("UPDATE telegram_users SET balance_sats = balance_sats + ? WHERE telegram_id=?")
          ->execute([$sats, $tid]);
    }

    $pdo->commit();
    echo json_encode(['ok'=>true, 'created'=>true, 'credited'=>(bool)$credited, 'delta_sats'=>$credited ? $sats : 0]);
    exit;
  }

  // déjà connu → mise à jour confs / statut
  $alreadyCredited = (int)$row['credited'] === 1;
  $prevConfs = (int)$row['confs'];

  // met à jour confs / status
  $newConfs = max($prevConfs, $confs);
  $newStatus = ($newConfs >= CONFIRM_TARGET) ? 'confirmed' : 'pending';

  $upd = $pdo->prepare("UPDATE btc_deposits SET confs=?, status=?, updated_at=? WHERE txid=? AND vout=?");
  $upd->execute([$newConfs, $newStatus, $now, $txid, $vout]);

  // si on passe le seuil de confs et pas encore crédité → créditer une seule fois
  if (!$alreadyCredited && $newConfs >= CONFIRM_TARGET) {
    $pdo->prepare("UPDATE btc_deposits SET credited=1, updated_at=? WHERE txid=? AND vout=?")
        ->execute([$now, $txid, $vout]);

    $pdo->prepare("UPDATE telegram_users SET balance_sats = balance_sats + ? WHERE telegram_id=?")
        ->execute([$sats, $tid]);

    $pdo->commit();
    echo json_encode(['ok'=>true, 'created'=>false, 'credited'=>true, 'delta_sats'=>$sats]);
    exit;
  }

  $pdo->commit();
  echo json_encode(['ok'=>true, 'created'=>false, 'credited'=>false, 'delta_sats'=>0, 'status'=>$newStatus, 'confs'=>$newConfs]);
} catch (Throwable $e) {
  try { $pdo?->rollBack(); } catch (Throwable $e2) {}
  http_response_code(500);
  echo json_encode(['ok'=>false, 'error'=>'server_error', 'detail'=>$e->getMessage()]);
}
