<?php
// api/index.php
declare(strict_types=1);
session_start();
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../includes/db.php'; // $pdo (PDO MySQL)

function fail(string $msg, int $code=400){
  http_response_code($code);
  echo json_encode(['ok'=>false,'error'=>$msg], JSON_UNESCAPED_UNICODE);
  exit;
}
function ok(array $data=[]){
  echo json_encode(['ok'=>true] + $data, JSON_UNESCAPED_UNICODE);
  exit;
}
function requireAuth($tid){ if(!$tid){ fail('Non authentifié.', 401); } }

// --- Supporte FormData ET JSON ---
$payload = $_POST;
if (empty($payload)) {
  $raw = file_get_contents('php://input');
  if ($raw) {
    $j = json_decode($raw, true);
    if (is_array($j)) $payload = $j;
  }
}
$action = $payload['action'] ?? '';

// --- Telegram ID : session OU payload (fallback propre) ---
$telegram_id = $_SESSION['telegram_id'] ?? null;
$payload_tid = $payload['telegram_id'] ?? $payload['tg_id'] ?? null;
if(!$telegram_id && $payload_tid){ $telegram_id = (int)$payload_tid; }

try{
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

  switch ($action) {

    /* ==================== PANIER ==================== */
    case 'cart.get': {
      requireAuth($telegram_id);
      $sql = "
        SELECT
          p.id AS product_id,
          p.nom,
          p.image,
          pa.quantite AS pack_qty,
          COUNT(*) AS units,
          COALESCE(pp.prix, 0) AS unit_price
        FROM paniers pa
        JOIN produits p ON p.id = pa.produit_id
        LEFT JOIN produit_prix pp
          ON pp.produit_id = p.id
         AND CAST(pp.quantite AS UNSIGNED) = pa.quantite
        WHERE pa.telegram_id = ?
        GROUP BY p.id, pa.quantite, p.nom, p.image, pp.prix
        ORDER BY p.nom ASC
      ";
      $st = $pdo->prepare($sql);
      $st->execute([$telegram_id]);
      $rows = $st->fetchAll(PDO::FETCH_ASSOC);

      $total = 0.0; $count = 0;
      foreach ($rows as &$r){
        $r['pack_qty']   = (int)$r['pack_qty'];
        $r['units']      = (int)$r['units'];
        $r['unit_price'] = (float)$r['unit_price'];
        $r['line_total'] = $r['units'] * $r['unit_price'];
        $total          += $r['line_total'];
        $count          += $r['units'];
      }
      ok(['items'=>$rows,'total'=>$total,'count'=>$count]);
    }

    case 'cart.add': {
      requireAuth($telegram_id);
      $pid  = (int)($payload['product_id'] ?? 0);
      $pack = (int)($payload['pack_qty'] ?? 0);
      if ($pid<=0 || $pack<=0) fail('Paramètres invalides.');
      $st = $pdo->prepare("INSERT INTO paniers(telegram_id, produit_id, quantite) VALUES (?, ?, ?)");
      $st->execute([$telegram_id, $pid, $pack]);
      ok();
    }

    case 'cart.qty': {
      requireAuth($telegram_id);
      $pid   = (int)($payload['product_id'] ?? 0);
      $pack  = (int)($payload['pack_qty'] ?? 0);
      $delta = (int)($payload['delta'] ?? 0);
      if ($pid<=0 || $pack<=0 || $delta==0) fail('Paramètres invalides.');

      if ($delta > 0) {
        $st = $pdo->prepare("INSERT INTO paniers(telegram_id, produit_id, quantite) VALUES (?, ?, ?)");
        for ($i=0; $i<$delta; $i++) $st->execute([$telegram_id, $pid, $pack]);
      } else {
        $n = -$delta;
        for ($i=0; $i<$n; $i++){
          $pdo->prepare("
            DELETE FROM paniers
            WHERE id = (
              SELECT id FROM (
                SELECT id FROM paniers
                 WHERE telegram_id=? AND produit_id=? AND quantite=?
                 ORDER BY date_ajout ASC LIMIT 1
              ) t
            )
          ")->execute([$telegram_id,$pid,$pack]);
        }
      }
      ok();
    }

    case 'cart.remove': {
      requireAuth($telegram_id);
      $pid  = (int)($payload['product_id'] ?? 0);
      $pack = (int)($payload['pack_qty'] ?? 0);
      if ($pid<=0 || $pack<=0) fail('Paramètres invalides.');
      $st = $pdo->prepare("DELETE FROM paniers WHERE telegram_id=? AND produit_id=? AND quantite=?");
      $st->execute([$telegram_id, $pid, $pack]);
      ok();
    }

    case 'cart.clear': {
      requireAuth($telegram_id);
      $pdo->prepare("DELETE FROM paniers WHERE telegram_id=?")->execute([$telegram_id]);
      ok();
    }

    /* ==================== ADRESSE ==================== */
    case 'address.get': {
  requireAuth($telegram_id);

  try {
    // Nouvelle version (avec ville + country)
    $st = $pdo->prepare("SELECT nom, prenom, adresse, code_postal, telephone, ville, country
                         FROM adresses WHERE telegram_id=? LIMIT 1");
    $st->execute([$telegram_id]);
  } catch (Throwable $e1) {
    try {
      // Ancien schéma (avec ville mais sans country)
      $st = $pdo->prepare("SELECT nom, prenom, adresse, code_postal, telephone, ville
                           FROM adresses WHERE telegram_id=? LIMIT 1");
      $st->execute([$telegram_id]);
    } catch (Throwable $e2) {
      // Très ancien schéma (sans ville/country)
      $st = $pdo->prepare("SELECT nom, prenom, adresse, code_postal, telephone
                           FROM adresses WHERE telegram_id=? LIMIT 1");
      $st->execute([$telegram_id]);
    }
  }

  $row = $st->fetch(PDO::FETCH_ASSOC);
  ok(['address' => $row ?: null]);
}


    case 'address.save': {
  requireAuth($telegram_id);

  // alias front: cp|code_postal et tel|telephone
  $nom        = trim((string)($payload['nom'] ?? ''));
  $prenom     = trim((string)($payload['prenom'] ?? ''));
  $adresse    = trim((string)($payload['adresse'] ?? ''));
  $codePostal = trim((string)($payload['code_postal'] ?? $payload['cp'] ?? ''));
  $telephone  = trim((string)($payload['telephone'] ?? $payload['tel'] ?? ''));
  $ville      = trim((string)($payload['ville'] ?? ''));
  $country    = trim((string)($payload['country'] ?? ''));

  $st = $pdo->prepare("SELECT id FROM adresses WHERE telegram_id=? LIMIT 1");
  $st->execute([$telegram_id]);
  $row = $st->fetch(PDO::FETCH_ASSOC);

  if ($row) {
    // UPDATE
    try {
      // Nouvelle version (avec ville + country)
      $upd = $pdo->prepare("UPDATE adresses
                            SET nom=?, prenom=?, adresse=?, code_postal=?, telephone=?, ville=?, country=?
                            WHERE id=?");
      $upd->execute([$nom,$prenom,$adresse,$codePostal,$telephone,$ville,$country,$row['id']]);
    } catch (Throwable $e1) {
      try {
        // Schéma sans country
        $upd = $pdo->prepare("UPDATE adresses
                              SET nom=?, prenom=?, adresse=?, code_postal=?, telephone=?, ville=?
                              WHERE id=?");
        $upd->execute([$nom,$prenom,$adresse,$codePostal,$telephone,$ville,$row['id']]);
      } catch (Throwable $e2) {
        // Schéma sans ville/country
        $upd = $pdo->prepare("UPDATE adresses
                              SET nom=?, prenom=?, adresse=?, code_postal=?, telephone=?
                              WHERE id=?");
        $upd->execute([$nom,$prenom,$adresse,$codePostal,$telephone,$row['id']]);
      }
    }
  } else {
    // INSERT
    try {
      // Nouvelle version (avec ville + country)
      $ins = $pdo->prepare("INSERT INTO adresses
                            (telegram_id, nom, prenom, adresse, code_postal, telephone, ville, country)
                            VALUES (?,?,?,?,?,?,?,?)");
      $ins->execute([$telegram_id,$nom,$prenom,$adresse,$codePostal,$telephone,$ville,$country]);
    } catch (Throwable $e1) {
      try {
        // Schéma sans country
        $ins = $pdo->prepare("INSERT INTO adresses
                              (telegram_id, nom, prenom, adresse, code_postal, telephone, ville)
                              VALUES (?,?,?,?,?,?,?)");
        $ins->execute([$telegram_id,$nom,$prenom,$adresse,$codePostal,$telephone,$ville]);
      } catch (Throwable $e2) {
        // Schéma sans ville/country
        $ins = $pdo->prepare("INSERT INTO adresses
                              (telegram_id, nom, prenom, adresse, code_postal, telephone)
                              VALUES (?,?,?,?,?,?)");
        $ins->execute([$telegram_id,$nom,$prenom,$adresse,$codePostal,$telephone]);
      }
    }
  }

  ok(['address'=>[
    'nom'=>$nom,
    'prenom'=>$prenom,
    'adresse'=>$adresse,
    'code_postal'=>$codePostal,
    'telephone'=>$telephone,
    'ville'=>$ville,
    'country'=>$country,
  ]]);
}


    /* ==================== VÉRIFICATION (SANS AUTH) ==================== */
    case 'verify.check': {
      $code = trim((string)($payload['code'] ?? ''));
      if ($code === '') fail('Code manquant.');

      // created_at + compteur dans la colonne verification_count
      $st = $pdo->prepare("
        SELECT
          id,
          code,
          strain,
          description,
          image_path,
          created_at,
          verification_count
        FROM codes
        WHERE code = ?
        LIMIT 1
      ");
      $st->execute([$code]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if (!$row) fail('Code introuvable.', 404);

      // incrémenter le compteur
      $pdo->prepare("UPDATE codes SET verification_count = COALESCE(verification_count,0) + 1 WHERE id = ?")
          ->execute([$row['id']]);
      $row['verification_count'] = (int)$row['verification_count'] + 1;

      ok(['code'=>$row]);
    }

    /* ==================== PRODUITS PAR CATÉGORIE (SPA) ==================== */
    case 'products.byCategory': {
      // pas d'auth obligatoire pour lister
      $cat = trim((string)($payload['categorie'] ?? $payload['category'] ?? ''));
      if ($cat === '') fail('Catégorie manquante.');

      // produits + prix minimum (si présents dans produit_prix)
      $sql = "
        SELECT
          p.id,
          p.nom,
          p.image,
          p.description,
          p.farm,
          p.categorie,
          MIN(pp.prix) AS min_price
        FROM produits p
        LEFT JOIN produit_prix pp
               ON pp.produit_id = p.id
        WHERE p.categorie = ?
        GROUP BY p.id, p.nom, p.image, p.description, p.farm, p.categorie
        ORDER BY p.created_at DESC, p.id DESC
      ";
      $st = $pdo->prepare($sql);
      $st->execute([$cat]);
      $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

      foreach ($rows as &$r){
        $r['min_price'] = $r['min_price'] !== null ? (float)$r['min_price'] : null;
      }
      ok(['items'=>$rows, 'categorie'=>$cat]);
    }

    /* ==================== COMMANDE ==================== */
    case 'order.create': {
      requireAuth($telegram_id);
      $st = $pdo->prepare("
        SELECT
          p.nom,
          pa.quantite AS pack_qty,
          COUNT(*) AS units,
          COALESCE(pp.prix,0) AS unit_price
        FROM paniers pa
        JOIN produits p ON p.id = pa.produit_id
        LEFT JOIN produit_prix pp
          ON pp.produit_id = p.id
         AND CAST(pp.quantite AS UNSIGNED) = pa.quantite
        WHERE pa.telegram_id = ?
        GROUP BY p.nom, pa.quantite, pp.prix
      ");
      $st->execute([$telegram_id]);
      $items = $st->fetchAll(PDO::FETCH_ASSOC);
      if (!$items) fail('Panier vide.', 400);

      $contenu = '';
      $total = 0.0;
      foreach ($items as $it){
        $units = (int)$it['units'];
        $price = (float)$it['unit_price'];
        $lineTotal = $units * $price;
        $total += $lineTotal;
        $contenu .= sprintf("%s (%sg) x%d - %s€\n",
          $it['nom'], (int)$it['pack_qty'], $units, number_format($lineTotal, 2, ',', ' '));
      }

      $ins = $pdo->prepare("INSERT INTO commandes(telegram_id, contenu, total, statut) VALUES (?, ?, ?, 'en attente')");
      $ins->execute([$telegram_id, $contenu, $total]);
      $orderId = (int)$pdo->lastInsertId();

      $pdo->prepare("DELETE FROM paniers WHERE telegram_id=?")->execute([$telegram_id]);
      ok(['order_id'=>$orderId, 'total'=>$total, 'contenu'=>$contenu]);
    }

case 'track.ping': {
  // PAS de requireAuth ici : on accepte les pings hors WebApp.
  // $telegram_id est rempli si l'entête Telegram est présent, sinon 0.
  $telegram_id = (int)($telegram_id ?? 0);

  $page       = trim((string)($payload['page'] ?? ''));
  $username   = trim((string)($payload['username'] ?? ''));
  $first_name = trim((string)($payload['first_name'] ?? ''));

  $now = date('Y-m-d H:i:s');
  $ip  = $_SERVER['HTTP_CF_CONNECTING_IP'] ?? ($_SERVER['REMOTE_ADDR'] ?? '');
  $ua  = $_SERVER['HTTP_USER_AGENT'] ?? '';

  if ($telegram_id) {
    // upsert par telegram_id
    $u = $pdo->prepare("UPDATE visiteurs
                        SET username=COALESCE(NULLIF(?,''), username),
                            first_name=COALESCE(NULLIF(?,''), first_name),
                            last_seen=?,
                            ip=COALESCE(NULLIF(?,''), ip),
                            user_agent=COALESCE(NULLIF(?,''), user_agent),
                            visite_date=?     -- historise aussi l’ancienne colonne
                        WHERE telegram_id=?");
    $u->execute([$username,$first_name,$now,$ip,$ua,$now,$telegram_id]);

    if ($u->rowCount() === 0) {
      $i = $pdo->prepare("INSERT INTO visiteurs(telegram_id,username,first_name,last_seen,ip,user_agent,visite_date)
                          VALUES (?,?,?,?,?,?,?)");
      $i->execute([$telegram_id,$username,$first_name,$now,$ip,$ua,$now]);
    }

    if ($page === 'index') {
      $x = $pdo->prepare("UPDATE visiteurs SET last_index_at=? WHERE telegram_id=?");
      $x->execute([$now, $telegram_id]);
    }
  } else {
    // fallback sans Telegram: on logge par IP
    $i = $pdo->prepare("INSERT INTO visiteurs(ip,user_agent,visite_date,last_seen)
                        VALUES (?,?,?,?)
                        ON DUPLICATE KEY UPDATE
                          user_agent=VALUES(user_agent),
                          visite_date=VALUES(visite_date),
                          last_seen=VALUES(last_seen)");
    // (si pas de clé unique sur ip, c’est juste un INSERT)
    try { $i->execute([$ip,$ua,$now,$now]); } catch(Throwable $e) {
      $pdo->prepare("INSERT INTO visiteurs(ip,user_agent,visite_date,last_seen) VALUES (?,?,?,?)")
          ->execute([$ip,$ua,$now,$now]);
    }
  }

  ok(['ok'=>true]);
}

case 'track.fetch': {
  // Pas d'auth dure : utilisé par le front public pour afficher la carte
  $limit = max(1, min(20, (int)($payload['limit'] ?? 5)));

  // en ligne = ping < 3 min (last_seen sinon visite_date)
  $online = $pdo->query("
    SELECT telegram_id, username, first_name,
           COALESCE(last_seen, visite_date) AS last_seen
    FROM visiteurs
    WHERE COALESCE(last_seen, visite_date) >= (NOW() - INTERVAL 3 MINUTE)
    ORDER BY COALESCE(last_seen, visite_date) DESC
    LIMIT 20
  ")->fetchAll(PDO::FETCH_ASSOC);

  // derniers passages sur l'index
  $stm = $pdo->prepare("
    SELECT telegram_id, username, first_name, last_index_at
    FROM visiteurs
    WHERE last_index_at IS NOT NULL
    ORDER BY last_index_at DESC
    LIMIT ?
  ");
  $stm->bindValue(1, $limit, PDO::PARAM_INT);
  $stm->execute();
  $recent = $stm->fetchAll(PDO::FETCH_ASSOC);

  ok([
    'count_online' => count($online),
    'online'       => $online,
    'recent_index' => $recent
  ]);
}


    /* ==================== USER (optionnel) ==================== */
    case 'user.info': {
      requireAuth($telegram_id);
      $st = $pdo->prepare("SELECT telegram_id, username, first_name, last_name, photo_url FROM telegram_users WHERE telegram_id=? LIMIT 1");
      $st->execute([$telegram_id]);
      $u = $st->fetch(PDO::FETCH_ASSOC) ?: [];
      ok(['user'=>$u]);
    }

    default: fail('Action inconnue.');
  }

} catch(Throwable $e){
  fail('Erreur: '.$e->getMessage(), 500);
}
