<?php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

require __DIR__ . '/config.php';
require __DIR__ . '/vendor/autoload.php';

use BitWasp\Bitcoin\Bitcoin;
use BitWasp\Bitcoin\Math\Math;
use BitWasp\Bitcoin\Network\NetworkFactory;
use BitWasp\Bitcoin\Key\Factory\HierarchicalKeyFactory;
use BitWasp\Bitcoin\Address\SegwitAddress;
use BitWasp\Bitcoin\Script\WitnessProgram;

// ---------- Inputs ----------
$input = json_decode(file_get_contents('php://input'), true) ?: $_POST;
$telegram_id = (int)($input['telegram_id'] ?? $_GET['telegram_id'] ?? 0);
if (!$telegram_id) {
  echo json_encode(['ok'=>false,'error'=>'missing telegram_id']); exit;
}

// ---------- Auth ----------
require_auth_melt($telegram_id);

// ---------- Pré-requis ----------
if (!defined('ACCOUNT_XPUB') || !ACCOUNT_XPUB) {
  echo json_encode(['ok'=>false,'error'=>'xpub_not_configured']); exit;
}

// ---------- Fallback BCMath si GMP absent ----------
if (!extension_loaded('gmp')) {
  if (!extension_loaded('bcmath')) {
    http_response_code(500);
    echo json_encode(['ok'=>false,'error'=>'math_ext_missing','detail'=>'Activez gmp ou bcmath dans PHP']); exit;
  }
  $bc = new \Mdanter\Ecc\Math\BcMath();
  $math = new Math($bc);
  Bitcoin::setMath($math);
}

/** Convertit ypub/zpub -> xpub (prefix 0x0488B21E) si nécessaire */
function slip132_to_xpub(string $ext): string {
  $p = substr($ext, 0, 4);
  try { $raw = \BitWasp\Bitcoin\Base58::decodeCheck($ext); }
  catch (\Throwable $e) { return $ext; }
  if ($p !== 'xpub') {
    $raw = "\x04\x88\xB2\x1E" . substr($raw, 4);
    return \BitWasp\Bitcoin\Base58::encodeCheck($raw);
  }
  return $ext;
}

/** Schéma DB minimum + curseur global + historique (compat avec ton schéma actuel) */
function ensure_schema(PDO $pdo): void {
  $pdo->exec("CREATE TABLE IF NOT EXISTS telegram_users (
    telegram_id BIGINT NOT NULL UNIQUE,
    username VARCHAR(255) NULL,
    first_name VARCHAR(255) NULL,
    last_name VARCHAR(255) NULL,
    photo_url TEXT NULL,
    created_at TIMESTAMP NULL,
    last_seen TIMESTAMP NULL,
    is_banned TINYINT(1) NOT NULL DEFAULT 0,
    balance_sats BIGINT NOT NULL DEFAULT 0,
    btc_index INT NOT NULL DEFAULT 0,
    btc_address VARCHAR(128) NULL
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

  // Curseur global
  $pdo->exec("CREATE TABLE IF NOT EXISTS btc_cursor (
    id TINYINT PRIMARY KEY,
    next_index INT NOT NULL
  ) ENGINE=InnoDB");
  $pdo->exec("INSERT IGNORE INTO btc_cursor (id, next_index) VALUES (1, 0)");

  // Historique (souple, sans contraintes uniques pour éviter conflits si doublons existants)
  $pdo->exec("CREATE TABLE IF NOT EXISTS btc_address_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    telegram_id BIGINT NOT NULL,
    address VARCHAR(128) NOT NULL,
    idx INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    KEY (telegram_id),
    KEY (idx),
    KEY (address)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}

try {
  $network = NetworkFactory::bitcoin();
  Bitcoin::setNetwork($network);

  $xpub = slip132_to_xpub(ACCOUNT_XPUB);
  $pdo  = pdo();
  ensure_schema($pdo);

  $hkf  = new HierarchicalKeyFactory();
  $acct = $hkf->fromExtended($xpub);

  // --- Transaction : garantit un index unique même en concurrence
  $pdo->beginTransaction();

  // 1) verrou utilisateur (pas de colonne id ici)
  $st = $pdo->prepare("SELECT btc_address, btc_index FROM telegram_users WHERE telegram_id=? FOR UPDATE");
  $st->execute([$telegram_id]);
  $user = $st->fetch(PDO::FETCH_ASSOC);

  if (!$user) {
    // crée la ligne si absente
    $pdo->prepare("INSERT IGNORE INTO telegram_users(telegram_id, btc_index) VALUES(?, 0)")
        ->execute([$telegram_id]);
    $user = ['btc_address' => null, 'btc_index' => 0];
  }

  // Idempotent : si déjà une adresse, renvoie-la
  if (!empty($user['btc_address'])) {
    $pdo->commit();
    echo json_encode(['ok'=>true,'address'=>$user['btc_address'],'index'=>(int)$user['btc_index'],'already'=>true]);
    exit;
  }

  // 2) verrou curseur global
  $cur = $pdo->query("SELECT next_index FROM btc_cursor WHERE id=1 FOR UPDATE")->fetch(PDO::FETCH_ASSOC);
  if (!$cur) { // init de secours
    $pdo->exec("INSERT IGNORE INTO btc_cursor (id, next_index) VALUES (1, 0)");
    $cur = ['next_index' => 0];
  }
  $index = (int)$cur['next_index'];

  // 3) cherche le prochain index qui donne une adresse non utilisée
  while (true) {
    // dérive m/84'/0'/0'/0/index
    $child   = $acct->derivePath("0/{$index}");
    $pub     = $child->getPublicKey();
    $hash160 = $pub->getPubKeyHash();           // 20 bytes
    $wp      = new WitnessProgram(0, $hash160); // version 0
    $addr    = (new SegwitAddress($wp))->getAddress($network); // "bc1..."

    // adresse déjà attribuée ?
    $chk = $pdo->prepare("SELECT 1 FROM telegram_users WHERE btc_address = ? LIMIT 1");
    $chk->execute([$addr]);
    if (!$chk->fetch()) {
      break; // adresse libre
    }
    $index++; // essaie l’index suivant
  }

  // 4) enregistre pour cet utilisateur
  $pdo->prepare("UPDATE telegram_users SET btc_address=?, btc_index=? WHERE telegram_id=?")
      ->execute([$addr, $index, $telegram_id]);

  // historise
  $pdo->prepare("INSERT INTO btc_address_history(telegram_id, address, idx) VALUES(?,?,?)")
      ->execute([$telegram_id, $addr, $index]);

  // avance le curseur au-delà de l’index utilisé
  $updCur = $pdo->prepare("UPDATE btc_cursor SET next_index = GREATEST(next_index, ?) + 1 WHERE id=1");
  $updCur->execute([$index]);

  $pdo->commit();

  echo json_encode(['ok'=>true,'address'=>$addr,'index'=>$index,'already'=>false]);

} catch (\Throwable $e) {
  try { $pdo?->rollBack(); } catch (\Throwable $e2) {}
  http_response_code(500);
  echo json_encode(['ok'=>false,'error'=>'server_error','detail'=>$e->getMessage()]);
}
