<?php
// includes/tracking_providers.php
if (!defined('TRACKING_PROVIDERS_LOADED')) define('TRACKING_PROVIDERS_LOADED', 1);

require_once __DIR__ . '/credentials.php';

/* ---------- HTTP helper ---------- */
function http_json(string $method, string $url, array $headers = [], $body = null, int $timeout = 25) {
  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_CONNECTTIMEOUT => 10,
    CURLOPT_TIMEOUT => $timeout,
    CURLOPT_CUSTOMREQUEST => strtoupper($method),
    CURLOPT_SSL_VERIFYPEER => true,
    CURLOPT_SSL_VERIFYHOST => 2,
  ]);
  if ($headers) curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
  if ($body !== null) {
    if (is_array($body)) {
      $payload = json_encode($body, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
      $headers[] = 'Content-Type: application/json';
      curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    } else {
      $payload = (string)$body;
    }
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
  }
  $raw = curl_exec($ch);
  $err = curl_error($ch);
  $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);
  return ['ok' => ($err==='' && $raw!==false), 'status'=>$code, 'raw'=>$raw, 'error'=>$err];
}

/* ---------- Normalisation ---------- */
function extract_latest_time($events): ?string {
  $best = null;
  if (is_array($events)) {
    foreach ($events as $ev) {
      $t = $ev['time'] ?? $ev['time_utc'] ?? $ev['datetime'] ?? $ev['date'] ?? $ev['status_date'] ?? null;
      if ($t && ($best===null || strcmp($t,$best) > 0)) $best = $t;
    }
  }
  return $best;
}
function norm_error(string $src, string $msg, $raw=null): array {
  return ['source'=>$src, 'status_code'=>'EXCEPTION', 'status_text'=>$msg, 'raw'=>$raw];
}
function norm_not_found(string $src, string $msg): array {
  return ['source'=>$src, 'status_code'=>'NOT_FOUND', 'status_text'=>$msg, 'raw'=>null];
}

/* ---------- 17TRACK v1 ---------- */
function map_17_status(string $status): string {
  $s = strtoupper(preg_replace('/\s+/', '', $status));
  if ($s === 'DELIVERED') return 'DELIVERED';
  if (in_array($s, ['INTRANSIT','PICKUP','OUTFORDELIVERY'])) return 'IN_TRANSIT';
  if ($s === 'NOTFOUND' || $s === 'EXPIRED') return 'NOT_FOUND';
  if (in_array($s, ['EXCEPTION','UNDELIVERED','RETURNED'])) return 'EXCEPTION';
  return 'UNKNOWN';
}
function seventeen_headers(): array {
  if (!defined('SEVENTEEN_API_KEY') || !SEVENTEEN_API_KEY) return [];
  return [
    '17token: ' . SEVENTEEN_API_KEY,
    'Content-Type: application/json',
    'Accept: application/json',
  ];
}
function track_with_17track(string $tracking): array {
  if (!defined('SEVENTEEN_API_KEY') || !SEVENTEEN_API_KEY) {
    return norm_error('17track', 'Missing SEVENTEEN_API_KEY');
  }
  $base = 'https://api.17track.net/track/v1';

  // 1) Register (ignore si déjà enregistré)
  @http_json('POST', $base.'/register', seventeen_headers(), [
    ['number' => $tracking]
  ]);

  // 2) Get info
  $res = http_json('POST', $base.'/gettrackinfo', seventeen_headers(), [
    ['number' => $tracking]
  ]);
  if (!$res['ok']) return norm_error('17track', 'HTTP error: '.$res['error'], $res);

  $j = json_decode($res['raw'], true);
  if (!$j || !is_array($j)) return norm_error('17track', 'Bad JSON');

  $d = $j['data'][0] ?? null;
  if (!$d) return norm_not_found('17track', 'No data');

  $statusRaw = $d['track']['latest_status'] ?? ($d['status'] ?? 'unknown');
  $events    = $d['track']['details'] ?? ($d['track']['origin_info']['trackinfo'] ?? []);
  $lastTime  = extract_latest_time($events);

  return [
    'source'       => '17track',
    'status_code'  => map_17_status((string)$statusRaw),
    'status_text'  => is_string($statusRaw) ? $statusRaw : json_encode($statusRaw, JSON_UNESCAPED_UNICODE),
    'last_event_at'=> $lastTime,
    'delivered_at' => (stripos((string)$statusRaw,'deliver')!==false ? $lastTime : null),
    'raw'          => $j,
  ];
}

/* ---------- Thailand Post ---------- */
function tp_get_auth_token(): array {
  if (!defined('TP_MEMBER_TOKEN') || !TP_MEMBER_TOKEN) {
    return ['ok'=>false,'error'=>'Missing TP_MEMBER_TOKEN'];
  }
  $res = http_json('POST', 'https://trackapi.thailandpost.co.th/post/api/v1/authenticate/token', [
    'Authorization: Token '.TP_MEMBER_TOKEN,
    'Content-Type: application/json',
    'Accept: application/json'
  ], '{}');
  if (!$res['ok']) return ['ok'=>false,'error'=>'HTTP error: '.$res['error']];
  $j = json_decode($res['raw'], true);
  if (!$j || empty($j['token'])) return ['ok'=>false,'error'=>'Bad JSON or no token', 'raw'=>$res['raw'] ?? null];
  return ['ok'=>true,'token'=>$j['token']];
}
function map_tp_status(string $status): string {
  $s = strtoupper($status);
  if (strpos($s,'DELIVER')!==false) return 'DELIVERED';
  if (strpos($s,'CUSTOM')!==false || strpos($s,'HELD')!==false) return 'CUSTOMS';
  if (strpos($s,'TRANSIT')!==false || strpos($s,'IN TRANSIT')!==false || strpos($s,'ACCEPT')!==false || strpos($s,'ARRIVE')!==false) return 'IN_TRANSIT';
  if (strpos($s,'NOT FOUND')!==false) return 'NOT_FOUND';
  if (strpos($s,'EXCEPTION')!==false || strpos($s,'FAIL')!==false || strpos($s,'UNSUCCESS')!==false) return 'EXCEPTION';
  return 'UNKNOWN';
}
function track_with_thaipost(string $tracking): array {
  $auth = tp_get_auth_token();
  if (!$auth['ok']) return norm_error('thaipost','Auth error: '.$auth['error']);

  $res = http_json('POST', 'https://trackapi.thailandpost.co.th/post/api/v1/track', [
    'Authorization: Token '.$auth['token'],
    'Content-Type: application/json',
    'Accept: application/json'
  ], [
    'status'   => 'all',
    'language' => 'EN',
    'barcode'  => [$tracking],
  ]);
  if (!$res['ok']) return norm_error('thaipost','HTTP error: '.$res['error'], $res);

  $j = json_decode($res['raw'], true);
  if (!is_array($j)) return norm_error('thaipost','Bad JSON');

  $items = $j['response']['items'][$tracking] ?? null;
  if (!$items || !is_array($items) || !count($items)) return norm_not_found('thaipost','No items');

  $latest = end($items);
  $desc   = $latest['status_description'] ?? ($latest['status'] ?? 'Unknown');

  return [
    'source'       => 'thaipost',
    'status_code'  => map_tp_status((string)($latest['status'] ?? $desc)),
    'status_text'  => $desc,
    'last_event_at'=> $latest['status_date'] ?? null,
    'delivered_at' => (stripos($desc,'deliver')!==false ? ($latest['status_date'] ?? null) : null),
    'raw'          => $j,
  ];
}

/* ---------- (Optionnel) La Poste ---------- */
function track_with_laposte(string $tracking): array {
  if (!defined('LAPOSTE_API_KEY') || !LAPOSTE_API_KEY) {
    return norm_error('laposte','Missing LAPOSTE_API_KEY');
  }
  $url = 'https://api.laposte.fr/suivi/v2/idships/' . rawurlencode($tracking) . '?lang=en';
  $res = http_json('GET', $url, ['X-Okapi-Key: '.LAPOSTE_API_KEY, 'Accept: application/json']);
  if (!$res['ok']) return norm_error('laposte','HTTP error: '.$res['error'], $res);
  $j = json_decode($res['raw'], true);
  if (!is_array($j)) return norm_error('laposte','Bad JSON');

  $shipment = $j['shipment'] ?? [];
  $timeline = $shipment['timeline'] ?? [];
  $latest   = null; foreach ($timeline as $ev) { if (!empty($ev['date'])) $latest = $ev; }
  $label = $latest['shortLabel'] ?? ($shipment['event'] ?? 'Unknown');

  $code = 'UNKNOWN'; $S = strtoupper($label);
  if (strpos($S,'LIVR')!==false || strpos($S,'DELIVER')!==false) $code='DELIVERED';
  elseif (strpos($S,'DOUANE')!==false || strpos($S,'CUSTOM')!==false) $code='CUSTOMS';
  elseif (strpos($S,'EN COURS')!==false || strpos($S,'TRANSIT')!==false || strpos($S,'ACHEMIN')!==false) $code='IN_TRANSIT';
  elseif (strpos($S,'NON TROUV')!==false || strpos($S,'NOT FOUND')!==false) $code='NOT_FOUND';
  elseif (strpos($S,'ANOMAL')!==false || strpos($S,'EXCEPT')!==false || strpos($S,'ECHEC')!==false) $code='EXCEPTION';

  return [
    'source'       => 'laposte',
    'status_code'  => $code,
    'status_text'  => $label,
    'last_event_at'=> $latest['date'] ?? null,
    'delivered_at' => ($code==='DELIVERED' ? ($latest['date'] ?? null) : null),
    'raw'          => $j,
  ];
}

/* ---------- Dispatcher ---------- */
function track_dispatch(string $tracking, string $apiChoice): array {
  $api = strtolower($apiChoice);
  if ($api === 'thaipost') return track_with_thaipost($tracking);
  if ($api === '17track')  return track_with_17track($tracking);
  if ($api === 'laposte')  return track_with_laposte($tracking);
  if ($api === 'all') {
    $res = [
      '17' => track_with_17track($tracking),
      'tp' => track_with_thaipost($tracking),
      // 'lp' => track_with_laposte($tracking),
    ];
    $prio = ['DELIVERED'=>3,'IN_TRANSIT'=>2,'CUSTOMS'=>1,'NOT_FOUND'=>0,'EXCEPTION'=>0,'UNKNOWN'=>0,'PENDING'=>0];
    $best = null; $bestScore = -1;
    foreach ($res as $r) {
      $score = $prio[$r['status_code'] ?? 'UNKNOWN'] ?? 0;
      if ($score > $bestScore) { $best = $r; $bestScore = $score; }
    }
    $best = $best ?: ['source'=>'all','status_code'=>'UNKNOWN','status_text'=>'No data'];
    $best['raw_all'] = $res;
    return $best;
  }
  // auto
  $r = track_with_17track($tracking);
  if (($r['status_code'] ?? 'UNKNOWN') === 'NOT_FOUND' || ($r['status_code'] ?? 'UNKNOWN') === 'UNKNOWN') {
    $t = track_with_thaipost($tracking);
    if (($t['status_code'] ?? 'UNKNOWN') !== 'UNKNOWN') $r = $t;
  }
  return $r;
}