<?php
// telegram_orders.php — Gestion du bot & du template des messages de commande Telegram
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

/* ==========================
   Helpers
   ========================== */
function e($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

// Template par défaut basé exactement sur ton exemple
const ORDER_TG_DEFAULT_TEMPLATE = "🔥 Commande MeltLabz

{PRODUCT_LINES}{PROMO_LINE}👤 Client :
• @{CLIENT_USERNAME}
• Prénom : {CLIENT_FIRST_NAME}
• Nom : {CLIENT_LAST_NAME}
• ID Telegram : {CLIENT_ID}

🏠 Livraison :
• Noms & Prénoms : {SHIP_FULL_NAME}
• Adresse : {SHIP_ADDRESS}
• Ville : {SHIP_CITY}
• Code Postal : {SHIP_ZIP}
• Téléphone : {SHIP_PHONE}

💳 Paiement : {PAYMENT_LINE}

🧾 Total du panier : {TOTAL_CART} €
🧾 Total à payer : {TOTAL_PAY} €";

/**
 * Récupère (ou crée) la ligne de config #1
 */
function order_tg_get_settings(PDO $pdo): array {
    $st = $pdo->prepare("SELECT * FROM telegram_order_settings WHERE id = 1 LIMIT 1");
    $st->execute();
    $row = $st->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        $ins = $pdo->prepare("
            INSERT INTO telegram_order_settings (id, bot_token, chat_id, template, is_active)
            VALUES (1, '', '', :tpl, 1)
        ");
        $ins->execute([':tpl' => ORDER_TG_DEFAULT_TEMPLATE]);
        $st->execute();
        $row = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    return $row;
}

/**
 * Simple wrapper pour appeler l’API Telegram
 */
function order_tg_call_api(string $token, string $method, array $params = []): array {
    if (trim($token) === '') {
        return ['ok' => false, 'description' => 'Bot token vide'];
    }

    $url = 'https://api.telegram.org/bot' . rawurlencode($token) . '/' . $method;

    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => "Content-Type: application/x-www-form-urlencoded\r\n",
            'content' => http_build_query($params),
            'timeout' => 10,
        ],
    ];

    $ctx  = stream_context_create($opts);
    $resp = @file_get_contents($url, false, $ctx);

    if ($resp === false) {
        return ['ok' => false, 'description' => 'Erreur HTTP'];
    }

    $data = json_decode($resp, true);
    if (!is_array($data)) {
        return ['ok' => false, 'description' => 'Réponse JSON invalide'];
    }

    return $data;
}

/**
 * Remplace les placeholders {KEY} dans le template
 */
function order_tg_format_message(string $tpl, array $vars): string {
    $search  = [];
    $replace = [];
    foreach ($vars as $k => $v) {
        $search[]  = '{' . $k . '}';
        $replace[] = $v;
    }
    return str_replace($search, $replace, $tpl);
}

/**
 * Jeu de variables d'exemple (pour test + aperçu)
 */
function order_tg_sample_vars(): array {
    return [
        'PRODUCT_LINES'      => "📦 Produit : Black Patronus\n📏 Quantité : 10 G – 300,00 €\n\n",
        'PROMO_LINE'         => "💸 Code promo TEST : -50,00 €\n\n",
        'CLIENT_USERNAME'    => "MeltLabzUS",
        'CLIENT_FIRST_NAME'  => "MeltLabz",
        'CLIENT_LAST_NAME'   => "",
        'CLIENT_ID'          => "7048196988",
        'SHIP_FULL_NAME'     => "f",
        'SHIP_ADDRESS'       => "N/A",
        'SHIP_CITY'          => "N/A",
        'SHIP_ZIP'           => "N/A",
        'SHIP_PHONE'         => "",
        'PAYMENT_LINE'       => "Ce client a payé 5,54 € via son Wallet.",
        'TOTAL_CART'         => "250,00",
        'TOTAL_PAY'          => "244,46",
    ];
}

/* ==========================
   Traitement POST (save/test)
   ========================== */

$alert_msg  = null;
$alert_type = 'info';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action   = $_POST['action'] ?? 'save_settings';
    $token    = trim($_POST['bot_token'] ?? '');
    $chatId   = trim($_POST['chat_id'] ?? '');
    $template = (string)($_POST['template'] ?? ORDER_TG_DEFAULT_TEMPLATE);

    try {
        // On s'assure que la config existe
        $settings = order_tg_get_settings($pdo);

        // Update DB
        $st = $pdo->prepare("
            UPDATE telegram_order_settings
            SET bot_token = :token,
                chat_id   = :chat_id,
                template  = :tpl
            WHERE id = 1
        ");
        $st->execute([
            ':token'   => $token,
            ':chat_id' => $chatId,
            ':tpl'     => $template,
        ]);

        if ($action === 'test_message') {
            if ($token === '' || $chatId === '') {
                $alert_type = 'danger';
                $alert_msg  = "Bot token et Chat ID sont obligatoires pour le test.";
            } else {
                $vars = order_tg_sample_vars();
                $msg  = order_tg_format_message($template, $vars);

                $api = order_tg_call_api($token, 'sendMessage', [
                    'chat_id'    => $chatId,
                    'text'       => $msg,
                    'parse_mode' => 'Markdown',
                ]);

                if (!empty($api['ok'])) {
                    $alert_type = 'success';
                    $alert_msg  = "Message de test envoyé avec succès dans le chat $chatId ✅";
                } else {
                    $alert_type = 'danger';
                    $alert_msg  = "Erreur Telegram: " . ($api['description'] ?? 'inconnue');
                }
            }
        } else {
            $alert_type = 'success';
            $alert_msg  = "Paramètres sauvegardés avec succès.";
        }

    } catch (Throwable $e) {
        $alert_type = 'danger';
        $alert_msg  = "Erreur serveur: " . $e->getMessage();
    }
}

// Recharge les settings après éventuelle mise à jour
$settings = order_tg_get_settings($pdo);
$currentToken   = (string)($settings['bot_token'] ?? '');
$currentChatId  = (string)($settings['chat_id'] ?? '');
$currentTpl     = (string)($settings['template'] ?? ORDER_TG_DEFAULT_TEMPLATE);
$isActive       = !empty($settings['is_active']);

// Calcul de l'aperçu dynamique
$previewVars     = order_tg_sample_vars();
$previewMessage  = order_tg_format_message($currentTpl, $previewVars);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Commandes Telegram - MeltLabz</title>

  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="assets/responsive-fixes.css">

  <style>
    .tg-status-badge{
      font-size:.75rem;
      padding:.3rem .6rem;
      border-radius:999px;
    }
    .tg-status-badge.on{background:#16a34a;color:#fff;}
    .tg-status-badge.off{background:#4b5563;color:#e5e7eb;}

    .tg-template-help code{
      background:#111827;
      border-radius:4px;
      padding:2px 4px;
      color:#e5e7eb;
    }
    textarea#template{
      font-family: SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace;
      font-size: .9rem;
      white-space: pre;
    }
  </style>
</head>
<body class="hold-transition sidebar-mini">
<div class="wrapper">

  <!-- Navbar -->
  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item">
        <a class="nav-link" data-widget="pushmenu" href="#" role="button">
          <i class="fas fa-bars"></i>
        </a>
      </li>
      <li class="nav-item d-none d-sm-inline-block">
        <a class="nav-link">MeltLabz Admin</a>
      </li>
    </ul>
  </nav>

  <?php include 'sidebar.php'; ?>

  <div class="content-wrapper p-3">
    <div class="container-fluid">

      <div class="d-flex flex-wrap align-items-center mb-3" style="gap:10px">
        <h1 class="h4 m-0 flex-grow-1">Commandes Telegram</h1>
        <span class="tg-status-badge <?= $isActive ? 'on' : 'off' ?>">
          <?= $isActive ? 'Config active' : 'Config inactive' ?>
        </span>
      </div>

      <?php if ($alert_msg): ?>
        <div class="alert alert-<?= e($alert_type) ?>">
          <?= e($alert_msg) ?>
        </div>
      <?php endif; ?>

      <div class="row">
        <div class="col-lg-4 mb-3">
          <div class="card card-dark">
            <div class="card-body">
              <h5 class="card-title mb-3">Bot & Groupe de réception</h5>

              <form method="post" id="tg-orders-form">
                <input type="hidden" name="action" id="tg-form-action" value="save_settings">

                <div class="form-group">
                  <label>Bot token</label>
                  <input type="text" class="form-control" name="bot_token"
                         value="<?= e($currentToken) ?>"
                         placeholder="8058xxxxxxxx:XXXXXXXX">
                  <small class="text-muted">
                    Token donné par BotFather (peut être le même bot que pour le webhook).
                  </small>
                </div>

                <div class="form-group">
                  <label>Chat ID du groupe</label>
                  <input type="text" class="form-control" name="chat_id"
                         value="<?= e($currentChatId) ?>"
                         placeholder="-100xxxxxxxxxx">
                  <small class="text-muted">
                    Utilise /chatid dans ton bot pour récupérer l'ID (négatif pour un groupe).
                  </small>
                </div>

                <div class="form-group">
                  <label>Template du message de commande</label>
                  <textarea class="form-control" id="template" name="template" rows="16"><?= e($currentTpl) ?></textarea>
                </div>

                <div class="tg-template-help small mb-3 text-muted">
                  <p class="mb-1">Variables disponibles :</p>
                  <ul class="mb-1 pl-3">
                    <li><code>{PRODUCT_LINES}</code> — lignes des produits + quantités + prix</li>
                    <li><code>{PROMO_LINE}</code> — ligne du code promo (vide si aucun)</li>
                    <li><code>{CLIENT_USERNAME}</code>, <code>{CLIENT_FIRST_NAME}</code>, <code>{CLIENT_LAST_NAME}</code>, <code>{CLIENT_ID}</code></li>
                    <li><code>{SHIP_FULL_NAME}</code>, <code>{SHIP_ADDRESS}</code>, <code>{SHIP_CITY}</code>, <code>{SHIP_ZIP}</code>, <code>{SHIP_PHONE}</code></li>
                    <li><code>{PAYMENT_LINE}</code> — ex: "Ce client a payé 5,54 € via son Wallet."</li>
                    <li><code>{TOTAL_CART}</code> — total du panier en € (ex: 250,00)</li>
                    <li><code>{TOTAL_PAY}</code> — total restant à payer en € (ex: 244,46)</li>
                  </ul>
                  <p class="mb-0">
                    Tu peux changer le texte autour tant que tu laisses les <code>{VARIABLES}</code> intactes.
                  </p>
                </div>

                <div class="d-flex" style="gap:8px;">
                  <button type="submit" class="btn btn-primary flex-fill" onclick="document.getElementById('tg-form-action').value='save_settings';">
                    💾 Sauvegarder
                  </button>
                  <button type="submit" class="btn btn-outline-info flex-fill" onclick="document.getElementById('tg-form-action').value='test_message';">
                    🧪 Envoyer un test
                  </button>
                </div>

              </form>
            </div>
          </div>
        </div>

        <div class="col-lg-8 mb-3">
          <div class="card card-dark">
            <div class="card-body">
              <h5 class="card-title mb-2">
                Aperçu logique (exemple)
              </h5>
              <p class="text-muted small">
                Généré à partir du template actuel + des données d'exemple.  
                Le vrai message sera construit depuis <code>checkout.php</code> avec le panier réel.
              </p>
<pre class="bg-dark text-light p-3 rounded small mb-0" style="white-space:pre-wrap;"><?= e($previewMessage) ?></pre>
            </div>
          </div>
        </div>

      </div><!-- row -->

    </div>
  </div>

</div>

<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>
</body>
</html>