<?php
// promo_codes.php — Gestion des codes promotionnels (AJAX, sans rechargement)
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

/* ==========================
   Helpers
   ========================== */
function e($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

/**
 * Rendu du tableau des promos + pagination
 */
function render_promos_table(array $promos, array $namesBy, array $prodIdsBy, int $page, int $limit, int $total) {
    $pages = max(1, (int)ceil(max(0, $total) / max(1, $limit)));
    ?>
    <table class="table table-hover table-striped align-middle" id="promo-table">
      <thead class="thead-dark">
        <tr>
          <th style="width:60px">#</th>
          <th>Code</th>
          <th>Réduction</th>
          <th>Portée</th>
          <th>Min. commande</th>
          <th>Statut</th>
          <th>Créé le</th>
          <th class="text-right">Actions</th>
        </tr>
      </thead>
      <tbody>
      <?php if (!$promos): ?>
        <tr>
          <td colspan="8" class="text-center text-muted py-4">
            Aucun code promo pour l'instant. Clique sur <strong>“Nouveau code promo”</strong> pour en créer un ✨
          </td>
        </tr>
      <?php else: ?>
        <?php foreach ($promos as $row):
          $id      = (int)$row['id'];
          $code    = $row['code'];
          $desc    = $row['description'] ?? '';
          $dtype   = $row['discount_type'];
          $dvalue  = (float)$row['discount_value'];
          $apAll   = (int)$row['applies_to_all'];
          $minTot  = $row['min_order_total'];
          $isAct   = (int)$row['is_active'];
          $created = $row['created_at'];

          $scopeNames = $namesBy[$id] ?? [];
          $prodIds    = $prodIdsBy[$id] ?? [];

          // Format réduction
          if ($dtype === 'percent') {
              $labelDiscount = rtrim(rtrim((string)$dvalue, '0'), '.') . ' %';
              $badgeClass = 'badge-info';
          } else {
              $labelDiscount = number_format($dvalue, 2, ',', ' ') . ' €';
              $badgeClass = 'badge-success';
          }

          // Min commande
          if ($minTot !== null && $minTot !== '') {
              $minLabel = '≥ ' . number_format((float)$minTot, 2, ',', ' ') . ' €';
          } else {
              $minLabel = 'Aucun';
          }

          // Statut
          $statusLabel = $isAct ? 'Actif' : 'Inactif';
          $statusClass = $isAct ? 'badge-success' : 'badge-secondary';

          // Produits associés
          $dataProducts = htmlspecialchars(json_encode($prodIds), ENT_QUOTES, 'UTF-8');
          ?>
          <tr>
            <td data-label="#">
              <span class="text-muted">#<?= $id ?></span>
            </td>

            <td data-label="Code">
              <span class="font-weight-bold"><?= e($code) ?></span>
              <?php if ($desc !== ''): ?>
                <div class="text-muted small"><?= e($desc) ?></div>
              <?php endif; ?>
            </td>

            <td data-label="Réduction">
              <span class="badge <?= $badgeClass ?> px-2 py-1"><?= e($labelDiscount) ?></span>
              <div class="text-muted small"><?= $dtype === 'percent' ? 'Pourcentage' : 'Montant fixe' ?></div>
            </td>

            <td data-label="Portée">
              <?php if ($apAll): ?>
                <span class="badge badge-primary">Tous les produits</span>
              <?php elseif ($scopeNames): ?>
                <?php foreach ($scopeNames as $name): ?>
                  <span class="badge badge-info mb-1 d-inline-block"><?= e($name) ?></span>
                <?php endforeach; ?>
              <?php else: ?>
                <span class="text-muted small">Aucun produit sélectionné</span>
              <?php endif; ?>
            </td>

            <td data-label="Min. commande">
              <?= $minTot !== null && $minTot !== '' ? e($minLabel) : '<span class="text-muted">Aucun</span>' ?>
            </td>

            <td data-label="Statut">
              <span class="badge <?= $statusClass ?>"><?= e($statusLabel) ?></span>
            </td>

            <td data-label="Créé le">
              <?= e(date('d/m/Y H:i', strtotime($created))) ?>
            </td>

            <td class="text-right" data-label="Actions">
              <button type="button"
                      class="btn btn-sm btn-warning mb-1 btn-edit-promo"
                      data-id="<?= $id ?>"
                      data-code="<?= e($code) ?>"
                      data-description="<?= e($desc) ?>"
                      data-discount-type="<?= e($dtype) ?>"
                      data-discount-value="<?= e($dvalue) ?>"
                      data-applies-to-all="<?= $apAll ?>"
                      data-min-order-total="<?= e($minTot) ?>"
                      data-products='<?= $dataProducts ?>'>
                ✏️ Éditer
              </button>

              <button type="button"
                      class="btn btn-sm btn-secondary mb-1 btn-toggle-promo"
                      data-id="<?= $id ?>">
                <?= $isAct ? 'Désactiver' : 'Activer' ?>
              </button>

              <button type="button"
                      class="btn btn-sm btn-danger mb-1 btn-delete-promo"
                      data-id="<?= $id ?>">
                🗑️ Supprimer
              </button>
            </td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
      </tbody>
    </table>

    <?php if ($pages > 1): ?>
      <nav class="mt-2">
        <ul class="pagination pagination-sm justify-content-center">
          <?php for ($p = 1; $p <= $pages; $p++): ?>
            <li class="page-item <?= $p === $page ? 'active' : '' ?>">
              <a href="#" class="page-link" data-page="<?= $p ?>"><?= $p ?></a>
            </li>
          <?php endfor; ?>
        </ul>
      </nav>
    <?php endif;
}

/* ==========================
   Actions AJAX (POST)
   ========================== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json; charset=utf-8');

    $action = $_POST['action'] ?? '';
    $res = ['ok' => false, 'message' => 'Action inconnue.'];

    try {
        if ($action === 'create' || $action === 'update') {
            $id   = isset($_POST['id']) ? (int)$_POST['id'] : 0;
            $code = trim($_POST['code'] ?? '');
            $description = trim($_POST['description'] ?? '');
            $discountType = $_POST['discount_type'] ?? '';
            $discountValueRaw = str_replace(',', '.', (string)($_POST['discount_value'] ?? '0'));
            $discountValue = (float)$discountValueRaw;
            $appliesAll = isset($_POST['applies_to_all']) ? 1 : 0;
            $minOrderRaw = trim((string)($_POST['min_order_total'] ?? ''));
            $minOrder = $minOrderRaw !== '' ? (float)str_replace(',', '.', $minOrderRaw) : null;
            $isActive = isset($_POST['is_active']) ? 1 : 1; // par défaut actif

            if ($code === '') {
                $res['message'] = 'Le code promo est obligatoire.';
                echo json_encode($res);
                exit;
            }

            if (!in_array($discountType, ['percent', 'amount'], true)) {
                $res['message'] = 'Type de réduction invalide.';
                echo json_encode($res);
                exit;
            }

            if ($discountValue <= 0) {
                $res['message'] = 'La valeur de la réduction doit être supérieure à 0.';
                echo json_encode($res);
                exit;
            }

            if ($discountType === 'percent' && $discountValue > 100) {
                $res['message'] = 'Un pourcentage ne peut pas dépasser 100%.';
                echo json_encode($res);
                exit;
            }

            if ($minOrder !== null && $minOrder < 0) {
                $res['message'] = 'Le montant minimum ne peut pas être négatif.';
                echo json_encode($res);
                exit;
            }

            // Unicité du code
            if ($action === 'create') {
                $st = $pdo->prepare("SELECT COUNT(*) FROM promo_codes WHERE code = ?");
                $st->execute([$code]);
            } else {
                $st = $pdo->prepare("SELECT COUNT(*) FROM promo_codes WHERE code = ? AND id <> ?");
                $st->execute([$code, $id]);
            }
            if ($st->fetchColumn() > 0) {
                $res['message'] = 'Ce code promo existe déjà.';
                echo json_encode($res);
                exit;
            }

            // Produits associés
            $productIds = [];
            if (!$appliesAll && !empty($_POST['products']) && is_array($_POST['products'])) {
                foreach ($_POST['products'] as $pid) {
                    $pid = (int)$pid;
                    if ($pid > 0) {
                        $productIds[] = $pid;
                    }
                }
                $productIds = array_values(array_unique($productIds));
            }

            $minOrderDb = $minOrder !== null ? $minOrder : null;

            if ($action === 'create') {
                $st = $pdo->prepare("
                    INSERT INTO promo_codes
                      (code, description, discount_type, discount_value, applies_to_all, min_order_total, is_active)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                $st->execute([
                    $code,
                    $description,
                    $discountType,
                    $discountValue,
                    $appliesAll,
                    $minOrderDb,
                    $isActive
                ]);
                $promoId = (int)$pdo->lastInsertId();
            } else {
                if ($id <= 0) {
                    $res['message'] = 'ID manquant pour la mise à jour.';
                    echo json_encode($res);
                    exit;
                }
                $st = $pdo->prepare("
                    UPDATE promo_codes
                    SET code = ?, description = ?, discount_type = ?, discount_value = ?,
                        applies_to_all = ?, min_order_total = ?, is_active = ?
                    WHERE id = ?
                ");
                $st->execute([
                    $code,
                    $description,
                    $discountType,
                    $discountValue,
                    $appliesAll,
                    $minOrderDb,
                    $isActive,
                    $id
                ]);
                $promoId = $id;

                $pdo->prepare("DELETE FROM promo_code_produits WHERE promo_id = ?")->execute([$promoId]);
            }

            if (!$appliesAll && $productIds) {
                $stL = $pdo->prepare("INSERT INTO promo_code_produits (promo_id, produit_id) VALUES (?, ?)");
                foreach ($productIds as $pid) {
                    $stL->execute([$promoId, $pid]);
                }
            }

            $res['ok'] = true;
            $res['message'] = 'Code promo enregistré avec succès.';
        }
        elseif ($action === 'delete') {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) {
                $res['message'] = 'ID invalide.';
            } else {
                $pdo->prepare("DELETE FROM promo_code_produits WHERE promo_id = ?")->execute([$id]);
                $pdo->prepare("DELETE FROM promo_codes WHERE id = ?")->execute([$id]);
                $res['ok'] = true;
                $res['message'] = 'Code promo supprimé.';
            }
        }
        elseif ($action === 'toggle') {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) {
                $res['message'] = 'ID invalide.';
            } else {
                $pdo->prepare("
                    UPDATE promo_codes
                    SET is_active = CASE WHEN is_active = 1 THEN 0 ELSE 1 END
                    WHERE id = ?
                ")->execute([$id]);
                $res['ok'] = true;
                $res['message'] = 'Statut mis à jour.';
            }
        }
    } catch (Throwable $e) {
        $res['ok'] = false;
        $res['message'] = 'Erreur serveur : ' . $e->getMessage();
    }

    echo json_encode($res);
    exit;
}

/* ==========================
   Listing (GET)
   ========================== */
$search = trim($_GET['search'] ?? '');
$page   = max(1, (int)($_GET['page'] ?? 1));
$limit  = 20;
$offset = ($page - 1) * $limit;

$cond = '1';
$args = [];
if ($search !== '') {
    $cond .= " AND (code LIKE :s OR description LIKE :s)";
    $args[':s'] = '%' . $search . '%';
}

$stmtCount = $pdo->prepare("SELECT COUNT(*) FROM promo_codes WHERE $cond");
$stmtCount->execute($args);
$totalRows = (int)$stmtCount->fetchColumn();

$sql = "SELECT * FROM promo_codes WHERE $cond ORDER BY created_at DESC LIMIT :lim OFFSET :off";
$stmt = $pdo->prepare($sql);
foreach ($args as $k => $v) {
    $stmt->bindValue($k, $v);
}
$stmt->bindValue(':lim', $limit, PDO::PARAM_INT);
$stmt->bindValue(':off', $offset, PDO::PARAM_INT);
$stmt->execute();
$promos = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Produits liés aux promos
$namesBy   = [];
$prodIdsBy = [];
if ($promos) {
    $ids = array_column($promos, 'id');
    $in  = implode(',', array_fill(0, count($ids), '?'));
    $stP = $pdo->prepare("
        SELECT pcp.promo_id, p.id AS produit_id, p.nom
        FROM promo_code_produits pcp
        JOIN produits p ON pcp.produit_id = p.id
        WHERE pcp.promo_id IN ($in)
        ORDER BY p.nom ASC
    ");
    $stP->execute($ids);
    foreach ($stP->fetchAll(PDO::FETCH_ASSOC) as $r) {
        $pidPromo = (int)$r['promo_id'];
        $namesBy[$pidPromo][]   = $r['nom'];
        $prodIdsBy[$pidPromo][] = (int)$r['produit_id'];
    }
}

// Liste de tous les produits pour la modale
$allProducts = [];
try {
    $stAll = $pdo->query("SELECT id, nom FROM produits ORDER BY nom ASC");
    $allProducts = $stAll->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    $allProducts = [];
}

// Mode AJAX (rafraîchissement du tableau uniquement)
if (isset($_GET['ajax']) && $_GET['ajax'] === '1') {
    ob_start();
    render_promos_table($promos, $namesBy, $prodIdsBy, $page, $limit, $totalRows);
    $html = ob_get_clean();
    echo $html;
    exit;
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Codes promotionnels - MeltLabz</title>

  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="assets/responsive-fixes.css">

  <style>
    /* Responsive "cards" sur mobile (comme produits.php) */
    @media (max-width: 860px){
      .table thead{display:none;}
      #promo-table{border-collapse:separate;border-spacing:0 12px}
      #promo-table tbody tr{
        display:block;
        background:#111827;
        border-radius:12px;
        padding:10px 12px;
        margin-bottom:10px;
        box-shadow:0 6px 18px rgba(0,0,0,.4);
      }
      #promo-table tbody tr td{
        display:flex;
        justify-content:space-between;
        align-items:center;
        border-top:none;
        padding:4px 0;
      }
      #promo-table tbody tr td::before{
        content:attr(data-label);
        font-weight:600;
        color:#6c7a89;
        margin-right:10px;
      }
      #promo-table tbody tr td:last-child{justify-content:flex-end;}
    }

    #btn-new-promo{
      background:linear-gradient(135deg,#6366f1,#ec4899);
      border:none;
      color:#fff;
      box-shadow:0 6px 16px rgba(99,102,241,.45);
      transition:transform .12s ease, box-shadow .12s ease, opacity .12s ease;
    }
    #btn-new-promo:hover{
      transform:translateY(-1px);
      box-shadow:0 10px 24px rgba(99,102,241,.6);
      opacity:.95;
    }

    .badge-info, .badge-success, .badge-primary{
      font-size:0.78rem;
    }
  </style>
</head>
<body class="hold-transition sidebar-mini">
<div class="wrapper">
  <!-- Navbar -->

  <nav class="main-header navbar navbar-expand navbar-white navbar-light">

    <ul class="navbar-nav">

      <li class="nav-item">

        <a class="nav-link" data-widget="pushmenu" href="#" role="button"><i class="fas fa-bars"></i></a>

      </li>

      <li class="nav-item d-none d-sm-inline-block"><a class="nav-link">MeltLabz Admin</a></li>

    </ul>

  </nav>
  <?php include 'sidebar.php'; ?>

  <div class="content-wrapper p-3">
    <div class="container-fluid">

      <div class="d-flex flex-wrap align-items-center mb-3" style="gap:10px">
        <h1 class="h4 m-0 flex-grow-1">Codes promotionnels</h1>

        <div class="input-group" style="max-width:320px">
          <div class="input-group-prepend">
            <span class="input-group-text">🔎</span>
          </div>
          <input type="text"
                 class="form-control"
                 id="promo-search"
                 placeholder="Rechercher un code ou une description…"
                 value="<?= e($search) ?>">
        </div>

        <button type="button" id="btn-new-promo" class="btn btn-sm rounded-pill px-3">
          ➕ Nouveau code promo
        </button>
      </div>

      <div class="alert alert-info small">
        <ul class="mb-0 pl-3">
          <li>Tu peux créer un code valable sur <strong>tout le catalogue</strong> ou seulement sur certains produits.</li>
          <li>Choisis entre une réduction en <strong>%</strong> ou en <strong>€</strong>.</li>
          <li>Option pour définir un <strong>montant minimum de commande</strong> avant que le code soit valide.</li>
        </ul>
      </div>

      <div class="card">
        <div class="card-body table-responsive" id="promo-results">
          <?php render_promos_table($promos, $namesBy, $prodIdsBy, $page, $limit, $totalRows); ?>
        </div>
      </div>

    </div>
  </div>
</div>

<!-- Modal création/édition -->
<div class="modal fade" id="promoModal" tabindex="-1" aria-labelledby="promoModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content bg-dark text-light">
      <div class="modal-header border-0">
        <h5 class="modal-title" id="promoModalLabel">Nouveau code promo</h5>
        <button type="button" class="close text-light" data-dismiss="modal" aria-label="Close">
          <span aria-hidden="true">&times;</span>
        </button>
      </div>
      <form id="promoForm">
        <div class="modal-body">
          <div id="promoAlert" class="alert alert-danger small d-none"></div>

          <input type="hidden" name="id" id="promo-id">
          <input type="hidden" name="action" id="promo-action" value="create">

          <div class="form-row">
            <div class="form-group col-md-4">
              <label for="promo-code">Code</label>
              <input type="text" class="form-control" id="promo-code" name="code" required>
              <small class="form-text text-muted">Ex: SENZU10, WELCOME5…</small>
            </div>

            <div class="form-group col-md-4">
              <label for="promo-discount-type">Type de réduction</label>
              <select class="form-control" id="promo-discount-type" name="discount_type">
                <option value="percent">Pourcentage (%)</option>
                <option value="amount">Montant fixe (€)</option>
              </select>
            </div>

            <div class="form-group col-md-4">
              <label for="promo-discount-value">Valeur</label>
              <input type="number" step="0.01" min="0" class="form-control" id="promo-discount-value" name="discount_value" required>
              <small class="form-text text-muted">% ou € selon le type.</small>
            </div>
          </div>

          <div class="form-group">
            <label for="promo-description">Description (optionnelle)</label>
            <textarea class="form-control" id="promo-description" name="description" rows="2" placeholder="Ex: -10% sur toutes les fleurs jusqu'à dimanche…"></textarea>
          </div>

          <div class="form-row align-items-center mb-3">
            <div class="form-group col-md-4 mb-2">
              <div class="custom-control custom-switch">
                <input type="checkbox" class="custom-control-input" id="promo-applies-all" name="applies_to_all" checked>
                <label class="custom-control-label" for="promo-applies-all">
                  Valable sur tous les produits
                </label>
              </div>
            </div>

            <div class="form-group col-md-4 mb-2">
              <label for="promo-min-order">Montant minimum de commande</label>
              <input type="number" step="0.01" min="0" class="form-control" id="promo-min-order" name="min_order_total" placeholder="ex: 50.00">
              <small class="form-text text-muted">Laisser vide pour aucun minimum.</small>
            </div>

            <div class="form-group col-md-4 mb-2">
              <div class="custom-control custom-switch mt-4">
                <input type="checkbox" class="custom-control-input" id="promo-is-active" name="is_active" checked>
                <label class="custom-control-label" for="promo-is-active">
                  Code actif
                </label>
              </div>
            </div>
          </div>

          <div class="form-group" id="promo-products-group" style="display:none;">
            <label for="promo-products">Limiter le code à certains produits</label>
            <select multiple class="form-control" id="promo-products" name="products[]"
                    size="<?= min(10, max(4, count($allProducts))) ?>">
              <?php foreach ($allProducts as $p): ?>
                <option value="<?= (int)$p['id'] ?>"><?= e($p['nom']) ?></option>
              <?php endforeach; ?>
            </select>
            <small class="form-text text-muted">Si aucun produit n'est sélectionné, le code ne sera appliqué à aucun produit.</small>
          </div>
        </div>

        <div class="modal-footer border-0">
          <button type="button" class="btn btn-secondary" data-dismiss="modal">Annuler</button>
          <button type="submit" class="btn btn-primary" id="promo-save-btn">Enregistrer</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>

<script>
$(function(){
  const $results = $('#promo-results');
  let typingTimer = null;

  function fetchPromos(page){
    const q = $('#promo-search').val() || '';
    $.get('promo_codes.php', { ajax: '1', search: q, page: page || 1 }, function(html){
      $results.fadeTo(120, 0.25, function(){
        $(this).html(html).fadeTo(150, 1);
      });
    });
  }

  $('#promo-search').on('input', function(){
    clearTimeout(typingTimer);
    typingTimer = setTimeout(function(){ fetchPromos(1); }, 250);
  });

  $(document).on('click', '#promo-results .pagination .page-link', function(e){
    e.preventDefault();
    const page = $(this).data('page') || $(this).text();
    fetchPromos(page);
  });

  function resetPromoForm(){
    const $form = $('#promoForm');
    $form[0].reset();
    $('#promo-id').val('');
    $('#promo-action').val('create');
    $('#promoModalLabel').text('Nouveau code promo');
    $('#promoAlert').addClass('d-none').text('');
    $('#promo-products').val([]);
    $('#promo-applies-all').prop('checked', true);
    $('#promo-products-group').hide();
    $('#promo-is-active').prop('checked', true);
  }

  $('#btn-new-promo').on('click', function(){
    resetPromoForm();
    $('#promoModal').modal('show');
  });

  $(document).on('change', '#promo-applies-all', function(){
    if (this.checked) {
      $('#promo-products-group').slideUp(150);
    } else {
      $('#promo-products-group').slideDown(150);
    }
  });

  $(document).on('click', '.btn-edit-promo', function(){
    const $btn = $(this);
    resetPromoForm();

    const id      = $btn.data('id');
    const code    = $btn.data('code');
    const desc    = $btn.data('description');
    const dtype   = $btn.data('discount-type');
    const dval    = $btn.data('discount-value');
    const apAll   = parseInt($btn.data('applies-to-all'), 10) === 1;
    const minOrd  = $btn.data('min-order-total');

    $('#promo-id').val(id);
    $('#promo-action').val('update');
    $('#promoModalLabel').text('Éditer le code promo #' + id);
    $('#promo-code').val(code);
    $('#promo-description').val(desc);
    $('#promo-discount-type').val(dtype);
    $('#promo-discount-value').val(dval);

    if (minOrd !== '' && minOrd !== null && minOrd !== undefined) {
      $('#promo-min-order').val(minOrd);
    }

    $('#promo-applies-all').prop('checked', apAll);
    if (apAll) {
      $('#promo-products-group').hide();
    } else {
      $('#promo-products-group').show();
    }

    let products = [];
    try {
      products = JSON.parse($btn.attr('data-products') || '[]');
    } catch(e) {
      products = [];
    }
    $('#promo-products').val(products);

    // Statut : si badge Actif → coché, sinon non
    const statusText = $btn.closest('tr').find('td[data-label="Statut"] .badge').text().trim().toLowerCase();
    $('#promo-is-active').prop('checked', statusText === 'actif');

    $('#promoModal').modal('show');
  });

  $('#promoForm').on('submit', function(e){
    e.preventDefault();
    const $btn = $('#promo-save-btn');
    const $alert = $('#promoAlert');

    $btn.prop('disabled', true).text('Enregistrement...');
    $alert.addClass('d-none').text('');

    $.post('promo_codes.php', $(this).serialize(), function(res){
      if (res && res.ok) {
        $('#promoModal').modal('hide');
        fetchPromos();
      } else {
        $alert.removeClass('d-none').text(res && res.message ? res.message : 'Erreur inconnue.');
      }
    }, 'json').fail(function(){
      $alert.removeClass('d-none').text('Erreur réseau ou serveur.');
    }).always(function(){
      $btn.prop('disabled', false).text('Enregistrer');
    });
  });

  $(document).on('click', '.btn-toggle-promo', function(){
    const id = $(this).data('id');
    $.post('promo_codes.php', { action: 'toggle', id: id }, function(res){
      if (res && res.ok) {
        fetchPromos();
      } else {
        alert(res && res.message ? res.message : 'Erreur lors du changement de statut.');
      }
    }, 'json');
  });

  $(document).on('click', '.btn-delete-promo', function(){
    const id = $(this).data('id');
    if (!confirm('Supprimer le code promo #' + id + ' ?')) return;
    $.post('promo_codes.php', { action: 'delete', id: id }, function(res){
      if (res && res.ok) {
        fetchPromos();
      } else {
        alert(res && res.message ? res.message : 'Erreur lors de la suppression.');
      }
    }, 'json');
  });

});
</script>
</body>
</html>