<?php
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

/* ===== Helpers ===== */
function e($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function trunc($s, $len=140){
  $s = trim((string)$s);
  if ($s === '') return '';
  $s = preg_replace('/\s+/', ' ', $s);
  if (mb_strlen($s,'UTF-8') <= $len) return $s;
  $out = mb_substr($s, 0, $len-1, 'UTF-8');
  $sp  = mb_strrpos($out, ' ', 0, 'UTF-8');
  if ($sp !== false && $sp > $len*0.6) $out = mb_substr($out, 0, $sp, 'UTF-8');
  return rtrim($out).'…';
}

/* ===== Params ===== */
$page     = max(1, (int)($_GET['page'] ?? 1));
$perPage  = (int)($_GET['perPage'] ?? 10);
if (!in_array($perPage, [10,20,30,50,100], true)) $perPage = 10;
$search   = trim((string)($_GET['search'] ?? ''));
$offset   = ($page - 1) * $perPage;

/* ===== WHERE ===== */
$where = '';
$args  = [];
if ($search !== '') {
  $where = "WHERE (LOWER(nom) LIKE :q OR LOWER(farm) LIKE :q OR LOWER(categorie) LIKE :q OR LOWER(description) LIKE :q)";
  $args[':q'] = '%'.mb_strtolower($search, 'UTF-8').'%';
}

/* ===== Count ===== */
$stC = $pdo->prepare("SELECT COUNT(*) FROM produits $where");
$stC->execute($args);
$total = (int)$stC->fetchColumn();

/* ===== Page data ===== */
$sql = "SELECT id, nom, image, categorie, farm, description, created_at
        FROM produits
        $where
        ORDER BY created_at DESC
        LIMIT :lim OFFSET :off";
$st = $pdo->prepare($sql);
foreach ($args as $k=>$v){ $st->bindValue($k, $v); }
$st->bindValue(':lim', $perPage, PDO::PARAM_INT);
$st->bindValue(':off', $offset, PDO::PARAM_INT);
$st->execute();
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

/* ===== Prices for visible rows ===== */
$pricesBy = [];
if ($rows) {
  $ids = array_column($rows, 'id');
  $in  = implode(',', array_fill(0,count($ids),'?'));
  $stP = $pdo->prepare("SELECT produit_id, quantite, prix
                        FROM produit_prix
                        WHERE produit_id IN ($in)
                        ORDER BY quantite ASC");
  $stP->execute($ids);
  foreach ($stP->fetchAll(PDO::FETCH_ASSOC) as $r) {
    $pid = (int)$r['produit_id'];
    $q   = rtrim(rtrim((string)$r['quantite'],'0'),'.');
    $p   = number_format((float)$r['prix'], 2, ',', ' ');
    $pricesBy[$pid][] = "{$q}G | {$p} €";
  }
}

/* ===== Renderer (tbody + pager) ===== */
function render_table($rows, $pricesBy, $page, $perPage, $total){
  ob_start(); ?>
  <tbody id="prod-tbody">
  <?php if (!$rows): ?>
    <tr><td colspan="6" class="text-center text-muted py-4">Aucun produit.</td></tr>
  <?php else: foreach ($rows as $r): $pid=(int)$r['id']; ?>
    <tr>
      <td class="text-muted">#<?= $pid ?></td>

      <td data-label="Produit">
        <div class="d-flex align-items-center" style="gap:10px">
          <!-- on conserve TON lien d'image tel quel -->
          <img src="../<?= e($r['image'] ?? '') ?>" alt="" style="width:48px;height:48px;object-fit:cover;border-radius:10px;border:1px solid rgba(0,0,0,.08)">

          <div>
            <div class="font-weight-bold"><?= e($r['nom']) ?></div>
            <?php if (!empty($r['categorie'])): ?>
              <span class="badge badge-secondary"><?= e($r['categorie']) ?></span>
            <?php endif; ?>
          </div>
        </div>
      </td>

      <td class="text-muted" data-label="Description"><?= e(trunc($r['description'] ?? '', 140)) ?></td>

      <td data-label="Farm">
        <?php if(!empty($r['farm'])): ?>
          <span class="badge badge-info"><?= e($r['farm']) ?></span>
        <?php else: ?>
          <span class="text-muted">—</span>
        <?php endif; ?>
      </td>

      <td data-label="Prix">
        <?php if (!empty($pricesBy[$pid])): foreach ($pricesBy[$pid] as $pill): ?>
          <span class="badge badge-light border font-weight-bold mr-1 mb-1 d-inline-block"><?= e($pill) ?></span>
        <?php endforeach; else: ?>
          <span class="text-muted">—</span>
        <?php endif; ?>
      </td>

      <td class="text-right" data-label="Actions">
        <a class="btn btn-sm btn-warning" href="edit_product.php?id=<?= $pid ?>">✏️ Éditer</a>
        <a class="btn btn-sm btn-danger" href="delete_product.php?id=<?= $pid ?>" onclick="return confirm('Supprimer le produit #<?= $pid ?> ?')">🗑️ Supprimer</a>
      </td>
    </tr>
  <?php endforeach; endif; ?>
  </tbody>

  <?php
  $pages = (int)ceil(max(1,$total) / max(1,$perPage));
  if ($pages > 1):
    $start = max(1, $page-2);
    $end   = min($pages, $page+2);
    if ($end-$start < 4){ $end = min($pages, $start+4); $start = max(1, $end-4); }
  ?>
  <nav id="prod-pager" aria-label="Pagination" class="mt-2">
    <ul class="pagination justify-content-center mb-1">
      <li class="page-item <?= $page<=1?'disabled':'' ?>">
        <a class="page-link" href="#" data-page="<?= max(1,$page-1) ?>">&laquo;</a>
      </li>
      <?php for($i=$start;$i<=$end;$i++): ?>
        <li class="page-item <?= $i===$page?'active':'' ?>">
          <a class="page-link" href="#" data-page="<?= $i ?>"><?= $i ?></a>
        </li>
      <?php endfor; ?>
      <li class="page-item <?= $page>=$pages?'disabled':'' ?>">
        <a class="page-link" href="#" data-page="<?= min($pages,$page+1) ?>">&raquo;</a>
      </li>
    </ul>
    <div class="text-center small text-muted">
      <?= ($total>0? (($page-1)*$perPage + 1) : 0) ?>–<?= min($total, $page*$perPage) ?> / <?= $total ?>
    </div>
  </nav>
  <?php endif;

  return ob_get_clean();
}

/* ===== AJAX short-circuit ===== */
if (isset($_GET['ajax']) && $_GET['ajax'] === '1') {
  header('Content-Type: text/html; charset=utf-8');
  echo render_table($rows, $pricesBy, $page, $perPage, $total);
  exit;
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Produits</title>

  <!-- 🟡 Garde tes includes AdminLTE/CSS existants ici si tu en as déjà -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">

  <style>
    /* Responsive "cards" sur mobile, sans casser AdminLTE */
    @media (max-width: 860px){
      .table thead{display:none;}
      #products-table{border-collapse:separate;border-spacing:0 12px}
      #products-table tbody tr{background:#fff;border:1px solid #eaeef3;border-radius:12px;box-shadow:0 6px 18px rgba(0,0,0,.04)}
      #products-table tbody tr td{display:flex;justify-content:space-between;align-items:flex-start;border:0;padding:.55rem .8rem}
      #products-table tbody tr td::before{content:attr(data-label);font-weight:600;color:#6c7a89;margin-right:10px}
      #products-table tbody tr td:first-child{display:none} /* cache ID en mobile */
      #products-table tbody tr td:last-child{justify-content:flex-end}
    }
  </style>
</head>
<body class="hold-transition sidebar-mini">
<div class="wrapper">
  <!-- 🟡 Garde ici ton header/navbar/sidebar existants (include si tu veux) -->
  <?php include 'sidebar.php';  ?>

  <div class="content-wrapper p-3">
    <div class="container-fluid">
      <div class="d-flex flex-wrap align-items-center mb-3" style="gap:10px">
        <h1 class="h4 m-0 flex-grow-1">Produits</h1>

        <div class="input-group" style="max-width:340px">
          <div class="input-group-prepend"><span class="input-group-text">🔎</span></div>
          <input type="text" class="form-control" id="search" placeholder="Rechercher (nom, farm, catégorie, description)…" value="<?= e($search) ?>">
        </div>

        <select id="perPage" class="custom-select" style="width:auto">
          <?php foreach([10,20,30,50,100] as $n): ?>
            <option value="<?= $n ?>" <?= $perPage===$n?'selected':'' ?>><?= $n ?></option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="card">
        <div class="card-body table-responsive" id="results">
          <table class="table table-hover align-middle mb-2" id="products-table">
            <thead class="thead-dark">
              <tr>
                <th style="width:64px">#</th>
                <th>Produit</th>
                <th>Description</th>
                <th>Farm</th>
                <th>Prix</th>
                <th class="text-right">Actions</th>
              </tr>
            </thead>
            <?= render_table($rows, $pricesBy, $page, $perPage, $total) ?>
          </table>
        </div>
      </div>
    </div>
  </div>

  <!-- 🟡 Footer/AdminLTE existants si tu en as -->
</div>

<!-- JS (garde aussi tes includes existants) -->
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>

<script>
/* === Recherche & pagination AJAX intégrées (même page) === */
(function(){
  const $results = $('#results');
  let typingTimer;

  function fetchData(page){
    const q  = $('#search').val() || '';
    const pp = $('#perPage').val() || '10';
    const url = new URL(window.location.href);
    url.searchParams.set('ajax', '1');
    url.searchParams.set('search', q);
    url.searchParams.set('perPage', pp);
    url.searchParams.set('page', page || 1);

    $.get(url.toString(), function(html){
      // Remplace tbody + pager uniquement (garde le reste intact)
      const $tmp    = $('<div>').html(html);
      const $tbody  = $tmp.find('#prod-tbody');
      const $pager  = $tmp.find('#prod-pager');

      const $oldTbd = $('#prod-tbody');
      if ($tbody.length && $oldTbd.length) $oldTbd.replaceWith($tbody);

      const $oldPg  = $('#prod-pager');
      if ($oldPg.length && $pager.length) $oldPg.replaceWith($pager);
      else if (!$oldPg.length && $pager.length) $results.append($pager);
    });
  }

  $('#search').on('input', function(){
    clearTimeout(typingTimer);
    typingTimer = setTimeout(()=>fetchData(1), 200);
  });

  $('#perPage').on('change', function(){ fetchData(1); });

  $(document).on('click', '#prod-pager .page-link', function(e){
    e.preventDefault();
    const page = $(this).data('page') || $(this).text();
    fetchData(page);
  });
})();
</script>
</body>
</html>
