<?php
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

// --- Guard & fetch ----------------------------------------------------------
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: codes.php');
    exit();
}
$id = (int)$_GET['id'];

$stmt = $pdo->prepare("SELECT * FROM codes WHERE id = ?");
$stmt->execute([$id]);
$code = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$code) {
    echo "Code introuvable.";
    exit();
}

$error = null;
$success = null;

// --- Update handler ---------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $new_code = strtoupper(trim($_POST['code'] ?? ''));
    $strain = trim($_POST['strain'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $remove_image = isset($_POST['remove_image']) ? 1 : 0;

    if ($new_code !== '' && $strain !== '') {

        // Base update values
        $image_path_to_save = $code['image_path'];

        // Handle image deletion (explicit)
        if ($remove_image && $image_path_to_save) {
            $absOld = dirname(__DIR__) . '/' . $image_path_to_save;
            if (is_file($absOld)) { @unlink($absOld); }
            $image_path_to_save = null;
        }

        // Handle new upload (if any)
        if (!empty($_FILES['image']['name'])) {
            $file = $_FILES['image'];

            if ($file['error'] !== UPLOAD_ERR_OK) {
                $error = "Erreur pendant l'upload (code {$file['error']}).";
            } else {
                // Validate mime
                $finfo = new finfo(FILEINFO_MIME_TYPE);
                $mime  = $finfo->file($file['tmp_name']);
                $allowed = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/webp' => 'webp', 'image/gif' => 'gif'];
                if (!isset($allowed[$mime])) {
                    $error = "Format d'image non supporté. Formats autorisés: JPG, PNG, WEBP, GIF.";
                } elseif ($file['size'] > 8 * 1024 * 1024) {
                    $error = "Fichier trop volumineux (max 8 Mo).";
                } else {
                    // Prepare destination
                    $root = dirname(__DIR__); // project root (parent of /admin)
                    $uploadRel = 'uploads/codes'; // will be referenced from the site root
                    $uploadAbs = $root . '/' . $uploadRel;

                    if (!is_dir($uploadAbs)) {
                        @mkdir($uploadAbs, 0755, true);
                    }

                    // Slug + unique filename
                    $base = preg_replace('~[^a-z0-9]+~i', '-', $new_code ?: ($code['code'] ?? 'code'));
                    $ext  = $allowed[$mime];
                    $destName = strtolower(trim($base, '-')) . '-' . $id . '-' . substr(sha1(uniqid('', true)), 0, 8) . '.' . $ext;
                    $destAbs  = $uploadAbs . '/' . $destName;
                    $destRel  = $uploadRel . '/' . $destName;

                    if (!move_uploaded_file($file['tmp_name'], $destAbs)) {
                        $error = "Impossible d'enregistrer l'image envoyée.";
                    } else {
                        // Delete previous file if any
                        if (!empty($code['image_path'])) {
                            $oldAbs = $root . '/' . $code['image_path'];
                            if (is_file($oldAbs)) { @unlink($oldAbs); }
                        }
                        $image_path_to_save = $destRel;
                    }
                }
            }
        }

        if (!$error) {
            $sql = "UPDATE codes
                       SET code = ?, strain = ?, description = ?, image_path = ?
                     WHERE id = ?";
            $st = $pdo->prepare($sql);
            $st->execute([$new_code, $strain, $description, $image_path_to_save, $id]);
            header("Location: codes.php?success=1");
            exit();
        }
    } else {
        $error = "Veuillez remplir tous les champs obligatoires (Code + Strain).";
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Modifier un Code</title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
  <style>
    .preview-wrap{display:flex;align-items:center;gap:16px;flex-wrap:wrap}
    .preview{
      width: 80px; height:80px; border-radius:10px; object-fit:cover; border:1px solid #e5e7eb;
      background:#f9fafb; display:inline-block;
    }
    .custom-file-input ~ .custom-file-label::after { content: "Parcourir"; }
  </style>
</head>
<body class="hold-transition sidebar-mini layout-fixed">
<div class="wrapper">
  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item">
        <a class="nav-link" data-widget="pushmenu" href="#" role="button">☰</a>
      </li>
      <li class="nav-item d-sm-none d-block">
        <span class="navbar-text">MeltLabz Admin</span>
      </li>
    </ul>
    <ul class="navbar-nav ml-auto">
      <li class="nav-item">
        <a href="codes.php" class="nav-link"><i class="fas fa-arrow-left mr-1"></i>Retour</a>
      </li>
    </ul>
  </nav>
<?php include 'sidebar.php'; ?>
  <div class="content-wrapper">
    <section class="content-header">
      <div class="container-fluid d-flex align-items-center justify-content-between">
        <h1 class="h3 mb-0">Modifier le code</h1>
      </div>
    </section>

    <section class="content">
      <div class="container-fluid">
        <?php if ($error): ?>
          <div class="alert alert-danger"><i class="fas fa-exclamation-triangle mr-1"></i><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>
        <?php if ($success): ?>
          <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
        <?php endif; ?>

        <div class="card">
          <div class="card-body">
            <form method="post" enctype="multipart/form-data">
              <div class="form-group">
                <label for="code">Code <span class="text-danger">*</span></label>
                <input type="text" name="code" id="code" class="form-control" value="<?= htmlspecialchars($code['code']) ?>" required>
              </div>

              <div class="form-group">
                <label for="strain">Strain <span class="text-danger">*</span></label>
                <input type="text" name="strain" id="strain" class="form-control" value="<?= htmlspecialchars($code['strain']) ?>" required>
              </div>

              <div class="form-group">
                <label for="description">Description</label>
                <textarea name="description" id="description" rows="3" class="form-control"><?= htmlspecialchars($code['description'] ?? '') ?></textarea>
              </div>

              <div class="form-group">
                <label>Image</label>
                <div class="preview-wrap mb-2">
                  <?php if (!empty($code['image_path'])): ?>
                    <img id="preview" class="preview" src="../<?= htmlspecialchars($code['image_path']) ?>" alt="Aperçu">
                  <?php else: ?>
                    <img id="preview" class="preview" src="https://via.placeholder.com/80?text=+" alt="Aperçu">
                  <?php endif; ?>
                  <?php if (!empty($code['image_path'])): ?>
                    <div class="form-check">
                      <input class="form-check-input" type="checkbox" value="1" id="remove_image" name="remove_image">
                      <label class="form-check-label" for="remove_image">Supprimer l'image actuelle</label>
                    </div>
                  <?php endif; ?>
                </div>
                <div class="custom-file">
                  <input type="file" class="custom-file-input" id="image" name="image" accept="image/*">
                  <label class="custom-file-label" for="image">Choisir une nouvelle image…</label>
                </div>
                <small class="form-text text-muted">Formats acceptés: JPG, PNG, WEBP, GIF • Max 8 Mo. L'image sera stockée dans <code>uploads/codes/</code>.</small>
              </div>

              <button type="submit" class="btn btn-primary">
                <i class="fas fa-save mr-1"></i> Enregistrer les modifications
              </button>
              <a href="codes.php" class="btn btn-secondary ml-2">Annuler</a>
            </form>
          </div>
        </div>
      </div>
    </section>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>
<script>
// Affiche le nom du fichier et un aperçu
$(function(){
  $('.custom-file-input').on('change', function(){
    let fileName = $(this).val().split('\\').pop();
    $(this).next('.custom-file-label').addClass('selected').html(fileName || 'Choisir une nouvelle image…');

    const input = this;
    if (input.files && input.files[0]) {
      const reader = new FileReader();
      reader.onload = function(e){ $('#preview').attr('src', e.target.result); };
      reader.readAsDataURL(input.files[0]);
    }
  });
});
</script>
</body>
</html>