<?php
// admin/index.php — Dashboard robuste (auto-détection + "en ligne" <= 4 min via PHP)
session_start();
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

/* ---------- Helpers SQL tolérants ----------- */
function pickColumn(PDO $pdo, string $table, array $cands, $fallback = null) {
  foreach ($cands as $c) {
    try { $pdo->query("SELECT `$c` FROM `$table` LIMIT 1"); return $c; } catch (Throwable $e) {}
  }
  return $fallback;
}
function safeQuery(PDO $pdo, string $sql, array $args = []) {
  try { $st = $pdo->prepare($sql); $st->execute($args); return $st->fetchAll(PDO::FETCH_ASSOC); }
  catch (Throwable $e) { return []; }
}
function kcol(array $rows, string $k) { return array_map(fn($r)=>$r[$k], $rows); }
function col_exists(PDO $pdo, string $table, string $col): bool {
  $sql = "SELECT 1
          FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME   = :table
            AND COLUMN_NAME  = :col
          LIMIT 1";
  $st = $pdo->prepare($sql);
  $st->execute([':table'=>$table, ':col'=>$col]);
  return (bool)$st->fetchColumn();
}

/* ---------- KPI simples ----------- */
$total_codes     = (int)(safeQuery($pdo, "SELECT COUNT(*) AS n FROM codes")[0]['n'] ?? 0);
$total_visiteurs = (int)(safeQuery($pdo, "SELECT COUNT(*) AS n FROM visiteurs")[0]['n'] ?? 0);
$total_produits  = (int)(safeQuery($pdo, "SELECT COUNT(*) AS n FROM produits")[0]['n'] ?? 0);
$total_clients   = (int)(safeQuery($pdo, "SELECT COUNT(*) AS n FROM telegram_users")[0]['n'] ?? 0);

/* ---------- Derniers clients ----------- */
$colUserCreated = pickColumn($pdo, 'telegram_users', ['created_at','createdAt','date','joined_at','reg_date'], 'created_at');
$recent_clients = safeQuery($pdo, "SELECT telegram_id, username, first_name, last_name, `$colUserCreated` AS created_at
                                   FROM telegram_users ORDER BY `$colUserCreated` DESC LIMIT 5");

/* ---------- Visiteurs / jour (30j) ----------- */
$colVisDate = pickColumn($pdo, 'visiteurs', ['visite_date','created_at','date','ts'], 'visite_date');
$vis_rows = safeQuery($pdo, "
  SELECT DATE(`$colVisDate`) AS d, COUNT(*) AS c
  FROM visiteurs
  WHERE `$colVisDate` >= (CURRENT_DATE - INTERVAL 29 DAY)
  GROUP BY DATE(`$colVisDate`)
  ORDER BY d
");
$daily_labels = []; $daily_visites = [];
for ($i=29; $i>=0; $i--){
  $d = (new DateTime())->modify("-$i day")->format('Y-m-d');
  $daily_labels[] = $d;
  $found = 0;
  foreach ($vis_rows as $r) { if ($r['d']===$d){ $found=(int)$r['c']; break; } }
  $daily_visites[] = $found;
}

/* ---------- Vérifications (Top 10 codes) ----------- */
$verif_rows   = safeQuery($pdo, "SELECT code, COALESCE(verification_count,0) AS vc FROM codes ORDER BY vc DESC, code ASC LIMIT 10");
$verif_labels = kcol($verif_rows, 'code');
$verif_values = array_map('intval', kcol($verif_rows, 'vc'));
$total_verifs = array_sum($verif_values);

/* ---------- Wallet (dépôts réels) ----------- */
$wallet_sum   = safeQuery($pdo, "SELECT COALESCE(SUM(eur_total),0) AS s FROM wallet_tx WHERE direction='credit'");
$walletTotal  = (float)($wallet_sum[0]['s'] ?? 0.0);
$tx_rows = safeQuery($pdo, "
  SELECT DATE(created_at) AS d, COALESCE(SUM(eur_total),0) AS s
  FROM wallet_tx
  WHERE direction='credit'
    AND created_at >= (CURRENT_DATE - INTERVAL 29 DAY)
  GROUP BY DATE(created_at)
  ORDER BY d
");
$tx_labels = []; $tx_values = [];
for ($i=29; $i>=0; $i--){
  $d = (new DateTime())->modify("-$i day")->format('Y-m-d');
  $tx_labels[] = $d;
  $v = 0.0;
  foreach ($tx_rows as $r) { if ($r['d'] === $d) { $v = (float)$r['s']; break; } }
  $tx_values[] = $v;
}

/* ---------- Commandes (CA / jour) ----------- */
$colOrderAmt  = pickColumn($pdo, 'commandes', ['total_price','total','amount','montant'], 'total');
$colOrderDate = pickColumn($pdo, 'commandes', ['created_at','date_commande','date','ts','createdAt'], 'created_at');
$cmd_rows = safeQuery($pdo, "
  SELECT DATE(`$colOrderDate`) AS d, COALESCE(SUM(`$colOrderAmt`),0) AS s
  FROM commandes
  WHERE `$colOrderDate` >= (CURRENT_DATE - INTERVAL 29 DAY)
  GROUP BY DATE(`$colOrderDate`)
  ORDER BY d
");
$cmd_labels = []; $cmd_values = [];
for ($i=29; $i>=0; $i--){
  $d = (new DateTime())->modify("-$i day")->format('Y-m-d');
  $cmd_labels[] = $d;
  $v = 0.0;
  foreach ($cmd_rows as $r) { if ($r['d'] === $d) { $v = (float)$r['s']; break; } }
  $cmd_values[] = $v;
}

/* ---------- Utils rendu ---------- */
function human_time($str){
  $t = strtotime($str);
  if(!$t) return htmlspecialchars($str,ENT_QUOTES,'UTF-8');
  $d = time()-$t;
  if ($d<60)   return 'just now';
  if ($d<3600) return floor($d/60).'m';
  if ($d<86400)return floor($d/3600).'h';
  return date('Y-m-d H:i', $t);
}
function fmt_age(int $sec): string {
  if ($sec < 60)    return 'just now';
  if ($sec < 3600)  return floor($sec/60).'m';
  if ($sec < 86400) return floor($sec/3600).'h';
  return floor($sec/86400).'d';
}
/* DATETIME ou timestamp numérique -> epoch */
function ts_or_0($v): int {
  if ($v === null || $v === '') return 0;
  if (is_numeric($v)) return (int)$v;          // epoch (int/str)
  $t = strtotime((string)$v);                  // DATETIME -> epoch
  return $t ?: 0;
}

/* ---------- Fenêtre "online" (minutes) ---------- */
$ONLINE_WINDOW_MIN = 4;
$win  = max(1, (int)$ONLINE_WINDOW_MIN);

/* ---------- Online + 5 dernières visites (filtré côté PHP, fuseau cohérent) ---------- */
try{
  $secs = max(60, $win * 60); // fenêtre en secondes

  // Colonnes présentes (protection selon schéma réel)
  $selSeen = col_exists($pdo,'visiteurs','last_seen')      ? "`last_seen`"      : "NULL AS last_seen";
  $selIdx  = col_exists($pdo,'visiteurs','last_index_at')  ? "`last_index_at`"  : "NULL AS last_index_at";
  $selVis  = col_exists($pdo,'visiteurs','visite_date')    ? "`visite_date`"    : "NULL AS visite_date";

  // On récupère puis on filtre/ordonne côté PHP (pas de NOW() MySQL → pas de souci de fuseau)
  $rows = safeQuery($pdo, "
    SELECT telegram_id, username, first_name, ip,
           $selSeen, $selIdx, $selVis
    FROM visiteurs
  ");

  $now    = time();
  $online = [];

  foreach ($rows as $r) {
    $ls = ts_or_0($r['last_seen']     ?? null);
    $li = ts_or_0($r['last_index_at'] ?? null);
    $vd = ts_or_0($r['visite_date']   ?? null);
    $last_ts = max($ls, $li, $vd);

    if ($last_ts > 0 && ($now - $last_ts) <= $secs) {
      $r['age_sec'] = $now - $last_ts; // pour l'affichage "3m", "15s", etc.
      $online[] = $r;
    }
  }

  // Les plus récents en premier (age le plus petit)
  usort($online, function($a,$b){ return ($a['age_sec'] <=> $b['age_sec']); });
  // Limite d'affichage raisonnable
  if (count($online) > 50) $online = array_slice($online, 0, 50);

  // 5 dernières visites (préférence last_index_at si dispo)
  if (col_exists($pdo,'visiteurs','last_index_at')){
    $recent = $pdo->query("
      SELECT telegram_id, username, first_name, last_index_at
      FROM visiteurs
      WHERE last_index_at IS NOT NULL
      ORDER BY last_index_at DESC
      LIMIT 5
    ")->fetchAll(PDO::FETCH_ASSOC);
  } else {
    $recent = $pdo->query("
      SELECT telegram_id, username, first_name, visite_date AS last_index_at
      FROM visiteurs
      WHERE visite_date IS NOT NULL
      ORDER BY visite_date DESC
      LIMIT 5
    ")->fetchAll(PDO::FETCH_ASSOC);
  }
} catch(Throwable $e){
  $online = $recent = [];
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Dashboard — MeltLabz</title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
  <style>
    .small-box .icon{top:6px}
    .card-title{margin-bottom:0}
    .chart-card{border:1px solid rgba(0,0,0,.06);border-radius:14px;box-shadow:0 6px 18px rgba(0,0,0,.08)}
  </style>
</head>
<body class="hold-transition sidebar-mini layout-fixed">
<div class="wrapper">

  <!-- Navbar -->
  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item">
        <a class="nav-link" data-widget="pushmenu" href="#" role="button"><i class="fas fa-bars"></i></a>
      </li>
      <li class="nav-item d-none d-sm-inline-block"><a class="nav-link">MeltLabz Admin</a></li>
    </ul>
  </nav>

  <!-- Sidebar -->
  <?php include 'sidebar.php'; ?>

  <!-- Content Wrapper -->
  <div class="content-wrapper">
    <div class="content-header">
      <div class="container-fluid">
        <h1 class="m-0">Dashboard</h1>
      </div>
    </div>

    <section class="content">
      <div class="container-fluid">

        <!-- KPIs -->
        <div class="row">
          <div class="col-lg-2 col-4">
            <div class="small-box bg-info">
              <div class="inner">
                <h3><?= number_format($total_codes, 0, ',', ' ') ?></h3>
                <p>Codes</p>
              </div>
              <div class="icon"><i class="fas fa-qrcode"></i></div>
            </div>
          </div>
          <div class="col-lg-2 col-4">
            <div class="small-box bg-success">
              <div class="inner">
                <h3><?= number_format($total_produits, 0, ',', ' ') ?></h3>
                <p>Produits</p>
              </div>
              <div class="icon"><i class="fas fa-box-open"></i></div>
            </div>
          </div>
          <div class="col-lg-2 col-4">
            <div class="small-box bg-warning">
              <div class="inner">
                <h3><?= number_format($total_visiteurs, 0, ',', ' ') ?></h3>
                <p>Visiteurs</p>
              </div>
              <div class="icon"><i class="fas fa-user-clock"></i></div>
            </div>
          </div>
          <div class="col-lg-3 col-4">
            <div class="small-box bg-primary">
              <div class="inner">
                <h3><?= number_format($walletTotal, 2, ',', ' ') ?> €</h3>
                <p>👛 Melt Wallet — Dépôts totaux</p>
              </div>
              <div class="icon"><i class="fas fa-wallet"></i></div>
            </div>
          </div>
          <div class="col-lg-2 col-4">
            <div class="small-box bg-danger">
              <div class="inner">
                <h3><?= number_format($total_clients, 0, ',', ' ') ?></h3>
                <p>Clients Telegram</p>
              </div>
              <div class="icon"><i class="fab fa-telegram"></i></div>
            </div>
          </div>
        </div>

        <!-- Tracker admin -->
        <div class="row" id="mlz-tracker-admin">
          <div class="col-md-6">
            <div class="card card-success">
              <div class="card-header">
                <h3 class="card-title">
                  <i class="fas fa-signal"></i> En ligne !
                  <span class="badge bg-success"><?= (int)count($online) ?></span>
                </h3>
              </div>
              <div class="card-body p-0">
                <ul class="list-group list-group-flush">
                  <?php if (!$online): ?>
                    <li class="list-group-item text-muted">Personne en ligne.</li>
                  <?php else: foreach ($online as $r):
                    $name = $r['username'] ? '@'.$r['username']
                          : ($r['first_name'] ?: ($r['telegram_id'] ? '#'.$r['telegram_id'] : ($r['ip'] ?? '—')));
                    $time = fmt_age((int)($r['age_sec'] ?? 0));
                  ?>
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                      <span><?= htmlspecialchars($name,ENT_QUOTES,'UTF-8') ?></span>
                      <small class="text-muted"><?= htmlspecialchars($time,ENT_QUOTES,'UTF-8') ?></small>
                    </li>
                  <?php endforeach; endif; ?>
                </ul>
              </div>
            </div>
          </div>

          <div class="col-md-6">
            <div class="card card-info">
              <div class="card-header">
                <h3 class="card-title"><i class="fas fa-history"></i> 5 dernières visites</h3>
              </div>
              <div class="card-body p-0">
                <ul class="list-group list-group-flush">
                  <?php if (!$recent): ?>
                    <li class="list-group-item text-muted">Aucune visite récente pour le moment.</li>
                  <?php else: foreach ($recent as $r):
                    $name = $r['username'] ? '@'.$r['username']
                          : ($r['first_name'] ?: ($r['telegram_id'] ? '#'.$r['telegram_id'] : '—'));
                    $time = human_time($r['last_index_at'] ?? '');
                  ?>
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                      <span><?= htmlspecialchars($name,ENT_QUOTES,'UTF-8') ?></span>
                      <small class="text-muted"><?= htmlspecialchars($time,ENT_QUOTES,'UTF-8') ?></small>
                    </li>
                  <?php endforeach; endif; ?>
                </ul>
              </div>
            </div>
          </div>
        </div>

        <!-- Graphs ligne 1 -->
        <div class="row">
          <div class="col-md-6">
            <div class="card chart-card">
              <div class="card-header d-flex justify-content-between align-items-center">
                <h3 class="card-title">📊 Visiteurs (30 jours)</h3>
              </div>
              <div class="card-body">
                <canvas id="visitsChart" height="160"></canvas>
              </div>
            </div>
          </div>
          <div class="col-md-6">
            <div class="card chart-card">
              <div class="card-header d-flex justify-content-between align-items-center">
                <h3 class="card-title">📦 Vérifications — Top 10 codes</h3>
                <span class="badge badge-primary">Total: <?= (int)$total_verifs ?></span>
              </div>
              <div class="card-body">
                <canvas id="verifChart" height="160"></canvas>
              </div>
            </div>
          </div>
        </div>

        <!-- Graphs ligne 2 -->
        <div class="row">
          <div class="col-md-6">
            <div class="card chart-card">
              <div class="card-header"><h3 class="card-title">💳 Transactions (30 jours)</h3></div>
              <div class="card-body">
                <canvas id="txChart" height="160"></canvas>
              </div>
            </div>
          </div>
          <div class="col-md-6">
            <div class="card chart-card">
              <div class="card-header"><h3 class="card-title">🧾 Commandes (CA — 30 jours)</h3></div>
              <div class="card-body">
                <canvas id="ordersChart" height="160"></canvas>
              </div>
            </div>
          </div>
        </div>

        <!-- Derniers clients -->
        <div class="row">
          <div class="col-12">
            <div class="card">
              <div class="card-header"><h3 class="card-title">🆕 Derniers clients (Telegram)</h3></div>
              <div class="card-body table-responsive">
                <table class="table table-striped">
                  <thead class="thead-light">
                    <tr>
                      <th>Telegram ID</th>
                      <th>Username</th>
                      <th>Nom</th>
                      <th>Créé le</th>
                    </tr>
                  </thead>
                  <tbody>
                  <?php if (!$recent_clients): ?>
                    <tr><td colspan="4" class="text-center text-muted">Aucun client récent.</td></tr>
                  <?php else: foreach ($recent_clients as $c): ?>
                    <tr>
                      <td><?= htmlspecialchars($c['telegram_id'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                      <td><?= htmlspecialchars($c['username'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                      <td><?= htmlspecialchars(trim(($c['first_name'] ?? '').' '.($c['last_name'] ?? '')), ENT_QUOTES, 'UTF-8') ?></td>
                      <td><?= htmlspecialchars($c['created_at'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    </tr>
                  <?php endforeach; endif; ?>
                  </tbody>
                </table>
              </div>
            </div>
          </div>
        </div>

      </div><!-- /.container-fluid -->
    </section>
  </div><!-- /.content-wrapper -->

</div><!-- /.wrapper -->

<!-- Scripts -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
<script>
/* === Visiteurs (30 jours) === */
(() => {
  const ctx = document.getElementById('visitsChart');
  if (!ctx) return;
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: <?= json_encode($daily_labels) ?>,
      datasets: [{
        label: 'Visiteurs',
        data: <?= json_encode($daily_visites) ?>,
        borderColor: 'rgba(255,193,7,1)',
        backgroundColor: 'rgba(255,193,7,.12)',
        tension: .3,
        fill: true
      }]
    },
    options: { responsive:true, interaction:{mode:'index',intersect:false}, scales:{ y:{ beginAtZero:true, ticks:{ precision:0 } } } }
  });
})();

/* === Vérifications (Top codes) === */
(() => {
  const ctx = document.getElementById('verifChart');
  if (!ctx) return;
  new Chart(ctx, {
    type: 'bar',
    data: {
      labels: <?= json_encode($verif_labels, JSON_UNESCAPED_UNICODE) ?>,
      datasets: [{
        label: 'Vérifications',
        data: <?= json_encode($verif_values) ?>,
        backgroundColor: 'rgba(54,162,235,.3)',
        borderColor: 'rgba(54,162,235,1)',
        borderWidth: 1
      }]
    },
    options: { responsive:true, scales:{ y:{ beginAtZero:true, ticks:{ precision:0 } } } }
  });
})();

/* === Transactions (€ / jour, 30 jours) === */
(() => {
  const ctx = document.getElementById('txChart');
  if (!ctx) return;
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: <?= json_encode($tx_labels) ?>,
      datasets: [{
        label: '€ déposés',
        data: <?= json_encode($tx_values) ?>,
        borderColor: 'rgba(75,192,192,1)',
        backgroundColor: 'rgba(75,192,192,.15)',
        tension: .3,
        fill: true
      }]
    },
    options: { responsive:true, interaction:{mode:'index',intersect:false}, scales:{ y:{ beginAtZero:true } } }
  });
})();

/* === Commandes (CA / jour, 30 jours) === */
(() => {
  const ctx = document.getElementById('ordersChart');
  if (!ctx) return;
  new Chart(ctx, {
    type: 'line',
    data: {
      labels: <?= json_encode($cmd_labels) ?>,
      datasets: [{
        label: '€ commandes',
        data: <?= json_encode($cmd_values) ?>,
        borderColor: 'rgba(255,99,132,1)',
        backgroundColor: 'rgba(255,99,132,.15)',
        tension: .3,
        fill: true
      }]
    },
    options: { responsive:true, interaction:{mode:'index',intersect:false}, scales:{ y:{ beginAtZero:true } } }
  });
})();
</script>
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>
</body>
</html>
