<?php
/**
 * admin/generate.php
 * Génération manuelle d'une transaction de CRÉDIT (ajout de solde)
 * - Style AdminLTE / Bootstrap identique à tes pages admin
 * - Recherche utilisateur par username ou ID Telegram
 * - Saisie du montant en EUR ou en SATS (conversion auto via le même helper que clients.php)
 * - Mise à jour du solde (balance_sats) + insertion dans wallet_tx (direction='credit')
 * - Option : tracer aussi dans btc_deposits_test (note/rate/eur/sats)
 */

session_start();
require_once dirname(__DIR__) . '/includes/db.php';
if (file_exists(__DIR__.'/auth_guard.php')) require __DIR__ . '/auth_guard.php';
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
  header('Location: login.php'); exit;
}

/* ===== Helpers ===== */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function findBalanceColumn(PDO $pdo): ?string {
  $stmt = $pdo->query("SELECT column_name FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = 'telegram_users'");
  $cols = $stmt->fetchAll(PDO::FETCH_COLUMN) ?: [];
  foreach (['balance_sats','wallet_sats','btc_sats','btc_balance_sats','balance_btc_sats'] as $c) {
    if (in_array($c, $cols, true)) return $c;
  }
  return null;
}
function btc_eur_rate(): float {
  // même logique que clients.php: cache local 60s + fallback
  $cache = __DIR__.'/.btc_eur.cache.json';
  if (is_file($cache) && (time()-filemtime($cache) < 60)) {
    $j = json_decode(@file_get_contents($cache), true);
    if (($j['eur'] ?? 0) > 0) return (float)$j['eur'];
  }
  $eur = 0.0;
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
  $local = @file_get_contents($scheme.'://'.$host.'/api/rates.php');
  if ($local) {
    $js = json_decode($local, true);
    if (!empty($js['ok']) && !empty($js['btc_eur'])) $eur = (float)$js['btc_eur'];
  }
  if ($eur <= 0) {
    $co = @file_get_contents('https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=eur');
    if ($co) { $js = json_decode($co, true); $eur = (float)($js['bitcoin']['eur'] ?? 0); }
  }
  if ($eur > 0) @file_put_contents($cache, json_encode(['eur'=>$eur]));
  return max(0.0, $eur);
}
function eur_to_sats(float $eur, float $eurPerBtc): int {
  if ($eurPerBtc <= 0) return 0;
  return (int) round(($eur / $eurPerBtc) * 100_000_000);
}
function sats_to_eur(int $sats, float $eurPerBtc): float {
  if ($eurPerBtc <= 0) return 0.0;
  return round(($sats / 100_000_000) * $eurPerBtc, 2);
}

/* ===== API de recherche utilisateur (pour auto-complétion) ===== */
if (($_GET['action'] ?? '') === 'search_user') {
  $q = trim($_GET['q'] ?? '');
  $out = [];
  if ($q !== '') {
    // recherche sur username ou telegram_id
    $sql = "SELECT telegram_id, username, first_name, last_name, photo_url FROM telegram_users
            WHERE username LIKE :q OR CAST(telegram_id AS CHAR) LIKE :q
            ORDER BY username IS NULL, username ASC LIMIT 15";
    $st = $pdo->prepare($sql);
    $st->execute([':q'=>"%$q%"]);
    while ($r = $st->fetch(PDO::FETCH_ASSOC)) {
      $out[] = [
        'telegram_id' => (string)$r['telegram_id'],
        'username'    => $r['username'] ?? '',
        'name'        => trim(($r['first_name'] ?? '').' '.($r['last_name'] ?? '')),
        'photo_url'   => $r['photo_url'] ?? '',
      ];
    }
  }
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode(['ok'=>true,'items'=>$out]);
  exit;
}

/* ===== Traitement du POST (création crédit) ===== */
$eurRate = btc_eur_rate();
$flash = ['type'=>null,'msg'=>null];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $tg      = trim($_POST['telegram_id'] ?? '');
  $in_eur  = trim($_POST['amount_eur'] ?? '');
  $in_sats = trim($_POST['amount_sats'] ?? '');
  $tx_ref  = trim($_POST['tx_ref'] ?? '');
  $note    = trim($_POST['note'] ?? '');
  $also_test = isset($_POST['also_test']) ? 1 : 0;

  // parse montants (accepte virgule)
  $amount_eur = null;
  if ($in_eur !== '') {
    $s = str_replace([' ', "\u{202F}"], '', $in_eur);
    $s = str_replace(',', '.', $s);
    $amount_eur = round((float)$s, 2);
  }
  $amount_sats = null;
  if ($in_sats !== '') {
    $s = str_replace([' ', "\u{202F}"], '', $in_sats);
    $amount_sats = (int)$s;
  }

  if ($tg === '' || (!ctype_digit($tg) && $tg[0] !== '-')) {
    $flash = ['type'=>'danger','msg'=>"ID Telegram invalide."];
  } else {
    // calcule sats/eur manquants
    if ($amount_sats === null && $amount_eur !== null) {
      $amount_sats = eur_to_sats($amount_eur, $eurRate);
    }
    if ($amount_eur === null && $amount_sats !== null) {
      $amount_eur = sats_to_eur($amount_sats, $eurRate);
    }
    if (($amount_sats ?? 0) <= 0) {
      $flash = ['type'=>'danger','msg'=>"Montant invalide (sats <= 0)."];
    } else {
      try {
        $pdo->beginTransaction();

        // lock user + colonne de solde
        $balCol = findBalanceColumn($pdo);
        if (!$balCol) throw new RuntimeException("Colonne de solde introuvable dans telegram_users.");
        $st = $pdo->prepare("SELECT telegram_id, $balCol AS balance FROM telegram_users WHERE telegram_id = ? FOR UPDATE");
        $st->execute([$tg]);
        $user = $st->fetch(PDO::FETCH_ASSOC);
        if (!$user) throw new RuntimeException("Utilisateur introuvable.");

        // crédit solde
        $newBal = (int)$user['balance'] + (int)$amount_sats;
        $up = $pdo->prepare("UPDATE telegram_users SET $balCol = ? WHERE telegram_id = ?");
        $up->execute([$newBal, $tg]);

        // journal wallet_tx (credit)
        $ins = $pdo->prepare("INSERT INTO wallet_tx (telegram_id, direction, amount_sats, eur_total, eur_per_btc, ref)
                              VALUES (?,?,?,?,?,?)");
        $ins->execute([$tg, 'credit', (int)$amount_sats, (float)$amount_eur, (float)$eurRate, ($tx_ref !== '' ? $tx_ref : 'manual')]);

        // option : tracer dans btc_deposits_test
        if ($also_test) {
          // crée la table si absente (optionnel)
          $pdo->exec("CREATE TABLE IF NOT EXISTS btc_deposits_test (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            telegram_id BIGINT UNSIGNED NOT NULL,
            sats BIGINT UNSIGNED NOT NULL,
            eur DECIMAL(18,2) DEFAULT NULL,
            rate_eurbtc DECIMAL(18,2) DEFAULT NULL,
            note VARCHAR(255) DEFAULT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            KEY(telegram_id), KEY(created_at)
          ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

          $ist = $pdo->prepare("INSERT INTO btc_deposits_test (telegram_id, sats, eur, rate_eurbtc, note)
                                VALUES (?,?,?,?,?)");
          $ist->execute([$tg, (int)$amount_sats, (float)$amount_eur, (float)$eurRate, ($note !== '' ? $note : 'manual')]);
        }

        $pdo->commit();
        $flash = ['type'=>'success','msg'=>"Crédit effectué : ".number_format($amount_eur,2,',',' ')." € (".number_format($amount_sats,0,',',' ')." sats) sur ID $tg."];
      } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $flash = ['type'=>'danger','msg'=>"Erreur: ".$e->getMessage()];
      }
    }
  }
}

/* ===== Récents crédits (dernier 50) ===== */
$recent = [];
try {
  $st = $pdo->query("SELECT wt.*, u.username, u.photo_url
                     FROM wallet_tx wt
                     LEFT JOIN telegram_users u ON u.telegram_id = wt.telegram_id
                     WHERE wt.direction='credit'
                     ORDER BY wt.id DESC
                     LIMIT 50");
  $recent = $st->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) { /* ignore */ }

?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="utf-8">
  <title>Générer un crédit</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">

  <!-- AdminLTE / Bootstrap (même stack que tes pages) -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">

  <style>
    .avatar { width:36px; height:36px; border-radius:50%; object-fit:cover; }
    .muted { color:#6c757d; }
    .help { font-size: .9rem; color:#6c757d; }
    .small-mono { font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, monospace; font-size:12px; }
  </style>
</head>
<body class="hold-transition sidebar-mini layout-fixed">
<div class="wrapper">

  <!-- Topbar minimal -->
  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item"><a class="nav-link" data-widget="pushmenu" href="#" role="button">☰</a></li>
      <li class="nav-item d-sm-none d-block"><span class="navbar-text">MeltLabz Admin</span></li>
    </ul>
  </nav>

  <?php if (file_exists(__DIR__.'/sidebar.php')) include __DIR__.'/sidebar.php'; ?>

  <div class="content-wrapper">
    <div class="content-header">
      <div class="container-fluid">
        <h1 class="m-0">⚙️ Générer un crédit wallet</h1>
        <p class="text-muted mb-0">Taux BTC→€ courant : <strong><?= $eurRate>0 ? number_format($eurRate,2,',',' ').' € / BTC' : '—' ?></strong></p>
      </div>
    </div>

    <section class="content">
      <div class="container-fluid">

        <?php if ($flash['type']): ?>
          <div class="alert alert-<?= h($flash['type']) ?>"><?= h($flash['msg']) ?></div>
        <?php endif; ?>

        <div class="card">
          <div class="card-header"><h3 class="card-title">Formulaire de crédit</h3></div>
          <div class="card-body">
            <form method="post" autocomplete="off" id="credit-form">
              <div class="form-row">
                <div class="form-group col-md-6">
                  <label>ID Telegram</label>
                  <div class="input-group">
                    <input type="text" class="form-control" name="telegram_id" id="telegram_id" placeholder="Ex: 123456789" required>
                    <div class="input-group-append">
                      <button type="button" class="btn btn-outline-secondary" id="btnSearch"><i class="fa fa-search"></i></button>
                    </div>
                  </div>
                  <small class="help">Tu peux saisir directement l’ID, ou cliquer sur 🔎 et taper un username (@user).</small>
                  <div id="userResult" class="mt-2"></div>
                </div>

                <div class="form-group col-md-3">
                  <label>Montant (EUR)</label>
                  <input type="text" class="form-control" name="amount_eur" id="amount_eur" placeholder="Ex: 50,00">
                  <small class="help">Si tu remplis EUR, les SATS sont calculés automatiquement.</small>
                </div>

                <div class="form-group col-md-3">
                  <label>Montant (SATS)</label>
                  <input type="text" class="form-control" name="amount_sats" id="amount_sats" placeholder="Ex: 250000">
                  <small class="help">Si tu remplis SATS, l’EUR s’affiche en face.</small>
                </div>
              </div>

              <div class="form-row">
                <div class="form-group col-md-6">
                  <label>Référence / TXID (optionnel)</label>
                  <input type="text" class="form-control small-mono" name="tx_ref" id="tx_ref" placeholder="txid ou texte libre (sera stocké dans wallet_tx.ref)">
                </div>
                <div class="form-group col-md-6">
                  <label>Note (optionnel)</label>
                  <input type="text" class="form-control" name="note" id="note" placeholder="Visible dans btc_deposits_test si coché ci-dessous">
                </div>
              </div>

              <div class="form-group form-check">
                <input type="checkbox" class="form-check-input" id="also_test" name="also_test">
                <label class="form-check-label" for="also_test">Tracer aussi dans <code>btc_deposits_test</code> (utile pour tests)</label>
              </div>

              <button type="submit" class="btn btn-primary">✅ Créditer</button>
            </form>
          </div>
        </div>

        <div class="card">
          <div class="card-header"><h3 class="card-title">Derniers crédits (wallet_tx)</h3></div>
          <div class="card-body table-responsive p-0">
            <table class="table table-hover table-striped mb-0">
              <thead class="thead-dark">
                <tr>
                  <th>Avatar</th>
                  <th>@user / ID</th>
                  <th>EUR</th>
                  <th>SATS</th>
                  <th>Taux</th>
                  <th>Réf</th>
                  <th>Date</th>
                </tr>
              </thead>
              <tbody>
              <?php foreach ($recent as $r):
                $eur = (float)($r['eur_total'] ?? 0);
                $sats= (int)($r['amount_sats'] ?? 0);
                $rate= (float)($r['eur_per_btc'] ?? 0);
                $name = $r['username'] ? '@'.$r['username'] : '—';
                $photo= $r['photo_url'] ?? '';
              ?>
                <tr>
                  <td><?= $photo ? '<img src="'.h($photo).'" class="avatar" alt="">' : '<div class="avatar" style="background:#e9ecef;"></div>' ?></td>
                  <td><?= h($name) ?><br><span class="muted small">ID: <?= h($r['telegram_id']) ?></span></td>
                  <td><strong><?= number_format($eur,2,',',' ') ?> €</strong></td>
                  <td><?= number_format($sats,0,',',' ') ?> sats</td>
                  <td><?= $rate>0 ? number_format($rate,2,',',' ').' €/BTC' : '—' ?></td>
                  <td class="small-mono"><?= h($r['ref'] ?: '—') ?></td>
                  <td><?= h($r['created_at']) ?></td>
                </tr>
              <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        </div>

      </div>
    </section>
  </div>
</div>

<!-- JS -->
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>
<script>
(function(){
  const rate = <?= json_encode($eurRate) ?>;
  const eurInput  = document.getElementById('amount_eur');
  const satsInput = document.getElementById('amount_sats');

  function parseNum(s){
    if(!s) return NaN;
    s = (''+s).replace(/\s|\u202F/g,'').replace(',','.');
    return parseFloat(s);
  }
  function fmtEur(v){ return (Math.round(v*100)/100).toLocaleString('fr-FR',{style:'currency',currency:'EUR'}); }
  function updateFromEUR(){
    const eur = parseNum(eurInput.value);
    if (isNaN(eur) || eur <= 0 || rate <= 0) return;
    const sats = Math.round((eur / rate) * 100000000);
    satsInput.value = sats;
  }
  function updateFromSATS(){
    const sats = parseInt(satsInput.value||'0',10);
    if (!sats || sats <= 0 || rate <= 0) return;
    const eur = (sats/100000000) * rate;
    eurInput.value = eur.toFixed(2).replace('.',',');
  }
  eurInput.addEventListener('input', updateFromEUR);
  satsInput.addEventListener('input', updateFromSATS);

  // recherche utilisateur
  const btn = document.getElementById('btnSearch');
  const target = document.getElementById('userResult');
  btn.addEventListener('click', async ()=>{
    const q = prompt("Tape un @username ou un ID Telegram :","");
    if(!q){ return; }
    target.innerHTML = '<div class="text-muted">Recherche…</div>';
    const r = await fetch('?action=search_user&q='+encodeURIComponent(q));
    const j = await r.json();
    if(!j.ok || !j.items.length){ target.innerHTML = '<div class="text-danger">Aucun résultat</div>'; return; }
    target.innerHTML = j.items.map(u=>{
      const av = u.photo_url ? `<img src="${u.photo_url}" class="avatar mr-2">` : `<span class="avatar mr-2" style="background:#e9ecef;display:inline-block;"></span>`;
      const name = u.username ? '@'+u.username : (u.name||'');
      return `<a href="#" class="d-flex align-items-center mb-2 pick-user" data-id="${u.telegram_id}">
                ${av}<div><div><strong>${name||'Inconnu'}</strong></div><div class="muted small">ID: ${u.telegram_id}</div></div>
              </a>`;
    }).join('');
    target.querySelectorAll('.pick-user').forEach(a=>{
      a.addEventListener('click', (e)=>{ e.preventDefault(); document.getElementById('telegram_id').value = a.dataset.id; target.innerHTML=''; });
    });
  });
})();
</script>
</body>
</html>
