<?php
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
  header('Location: login.php');
  exit();
}

// ---- helpers -------------------------------------------------
function csrf_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function csrf_check(string $t): void {
  if (empty($_SESSION['csrf']) || !hash_equals($_SESSION['csrf'], $t)) {
    http_response_code(403); exit('Invalid CSRF');
  }
}
function sats_to_btc($sats): float { return ((int)$sats)/1e8; }
function btc_to_sats($btc): int {
  $btc = (float)str_replace(',', '.', (string)$btc);
  return (int) round($btc * 100_000_000);
}
/** Taux BTC→EUR (cache 60s) */
function btc_eur_rate(): float {
  $cache = __DIR__ . '/.btc_eur.cache.json';
  $now = time();
  if (is_file($cache) && ($now - filemtime($cache) < 60)) {
    $j = json_decode(@file_get_contents($cache), true);
    if (isset($j['eur']) && $j['eur'] > 0) return (float)$j['eur'];
  }
  $eur = 0.0;
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
  $local = @file_get_contents($scheme.'://'.$host.'/api/rates.php');
  if ($local) {
    $js = json_decode($local, true);
    if (!empty($js['ok']) && !empty($js['btc_eur'])) $eur = (float)$js['btc_eur'];
  }
  if ($eur <= 0) {
    $co = @file_get_contents('https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=eur');
    if ($co) {
      $js = json_decode($co, true);
      $eur = (float)($js['bitcoin']['eur'] ?? 0);
    }
  }
  if ($eur > 0) @file_put_contents($cache, json_encode(['eur'=>$eur]));
  return max(0.0, $eur);
}

// ---- load user -----------------------------------------------
$telegram_id = (int)($_GET['id'] ?? 0);
if (!$telegram_id) { header('Location: clients.php'); exit; }

$sql = "SELECT tu.*, a.nom, a.prenom, a.adresse, a.ville, a.code_postal, a.telephone
        FROM telegram_users tu
        LEFT JOIN adresses a ON a.telegram_id = tu.telegram_id
        WHERE tu.telegram_id = ?";
$stmt = $pdo->prepare($sql);
$stmt->execute([$telegram_id]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$user) { http_response_code(404); exit('Utilisateur introuvable'); }

$eurRate = btc_eur_rate();
$ok=''; $err='';

// ---- handle POST ---------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  csrf_check($_POST['csrf'] ?? '');

  try {
    if (isset($_POST['save_profile'])) {
      $prenom  = trim($_POST['prenom']  ?? '');
      $nom     = trim($_POST['nom']     ?? '');
      $adresse = trim($_POST['adresse'] ?? '');
      $ville   = trim($_POST['ville']   ?? '');
      $cp      = trim($_POST['code_postal'] ?? '');
      $tel     = trim($_POST['telephone']   ?? '');

      // upsert dans adresses
      $exists = $pdo->prepare("SELECT 1 FROM adresses WHERE telegram_id=?");
      $exists->execute([$telegram_id]);
      if ($exists->fetchColumn()) {
        $u = $pdo->prepare("UPDATE adresses SET prenom=?, nom=?, adresse=?, ville=?, code_postal=?, telephone=? WHERE telegram_id=?");
        $u->execute([$prenom,$nom,$adresse,$ville,$cp,$tel,$telegram_id]);
      } else {
        $i = $pdo->prepare("INSERT INTO adresses(telegram_id, prenom, nom, adresse, ville, code_postal, telephone) VALUES(?,?,?,?,?,?,?)");
        $i->execute([$telegram_id,$prenom,$nom,$adresse,$ville,$cp,$tel]);
      }
      $ok = "Profil mis à jour.";
    }

    if (isset($_POST['save_wallet'])) {
      $btc       = (string)($_POST['balance_btc'] ?? '0');
      $sats      = btc_to_sats($btc);
      $btc_addr  = trim($_POST['btc_address'] ?? '');

      $w = $pdo->prepare("UPDATE telegram_users SET balance_sats=?, btc_address=? WHERE telegram_id=?");
      $w->execute([$sats, ($btc_addr ?: null), $telegram_id]);

      $ok = "Wallet mis à jour.";
    }

    if (isset($_POST['reset_addr'])) {
      $pdo->prepare("UPDATE telegram_users SET btc_address=NULL WHERE telegram_id=?")->execute([$telegram_id]);
      $ok = "Adresse BTC réinitialisée (le client en régénérera une).";
    }

    // reloader user après update
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$telegram_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

  } catch (Throwable $e) {
    $err = $e->getMessage();
  }
}

// valeurs affichées
$btc = sats_to_btc($user['balance_sats'] ?? 0);
$eur = $eurRate>0 ? $btc*$eurRate : 0.0;
?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Modifier utilisateur #<?= htmlspecialchars($telegram_id) ?></title>

  <!-- Bootstrap 4 + AdminLTE -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">

  <style>
    .card-glass{background:rgba(255,255,255,.06);border:1px solid rgba(255,255,255,.12);backdrop-filter:blur(8px)}
    .badge-id{background:#eef0f4;color:#333;border:1px solid #dfe3ea;font-weight:600}
    .form-note{font-size:.9rem;opacity:.7}
    @media (max-width:576px){
      .btn-block-mobile{display:block;width:100%}
      .grid-gap-sm>*{margin-bottom:.5rem}
    }
  </style>
</head>
<body class="hold-transition sidebar-mini layout-fixed">
<div class="wrapper">

  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item">
        <a class="nav-link" data-widget="pushmenu" href="#" role="button">☰</a>
      </li>
      <li class="nav-item d-sm-none d-block">
        <span class="navbar-text">MeltLabz Admin</span>
      </li>
    </ul>
  </nav>

  <?php include 'sidebar.php'; ?>

  <div class="content-wrapper">
    <div class="content-header">
      <div class="container-fluid">
        <a href="clients.php" class="btn btn-link p-0 mb-2"><i class="fa fa-arrow-left"></i> Retour à la liste</a>

        <h1 class="m-0">Modifier l’utilisateur</h1>
        <p class="text-muted mb-2">
          Taux BTC→€ : <strong><?= $eurRate>0 ? number_format($eurRate,2,',',' ') . ' € / BTC' : '—' ?></strong>
        </p>

        <?php if ($ok): ?>
          <div class="alert alert-success"><?= htmlspecialchars($ok) ?></div>
        <?php endif; ?>
        <?php if ($err): ?>
          <div class="alert alert-danger"><?= htmlspecialchars($err) ?></div>
        <?php endif; ?>
      </div>
    </div>

    <div class="content">
      <div class="container-fluid">

        <!-- Carte identitée -->
        <div class="card card-glass mb-3">
          <div class="card-body d-flex align-items-center flex-wrap">
            <img src="<?= htmlspecialchars($user['photo_url']) ?>" class="rounded-circle mr-3 mb-2" width="64" height="64" alt="avatar">
            <div class="mr-auto">
              <div><strong>@<?= htmlspecialchars($user['username'] ?? '') ?></strong>
                <span class="badge badge-id ml-2">ID: <?= (int)$telegram_id ?></span>
                <?php if (!empty($user['is_banned'])): ?>
                  <span class="badge badge-danger ml-1">Banni</span>
                <?php endif; ?>
              </div>
              <div class="text-muted">Vu le : <?= htmlspecialchars($user['last_seen'] ?? '') ?></div>
            </div>
            <div class="grid-gap-sm">
              <?php if (!empty($user['is_banned'])): ?>
                <a class="btn btn-success btn-block-mobile" href="toggle_ban.php?id=<?= (int)$telegram_id ?>&action=unban">
                  <i class="fa fa-unlock"></i> Débannir
                </a>
              <?php else: ?>
                <a class="btn btn-danger btn-block-mobile" href="toggle_ban.php?id=<?= (int)$telegram_id ?>&action=ban" onclick="return confirm('Bannir cet utilisateur ?');">
                  <i class="fa fa-ban"></i> Bannir
                </a>
              <?php endif; ?>
            </div>
          </div>
        </div>

        <div class="row">
          <!-- Profil -->
          <div class="col-lg-6">
            <div class="card card-glass">
              <div class="card-header"><h3 class="card-title">Profil</h3></div>
              <div class="card-body">
                <form method="post" class="row">
                  <input type="hidden" name="csrf" value="<?= csrf_token() ?>">
                  <div class="form-group col-sm-6">
                    <label>Prénom</label>
                    <input class="form-control" name="prenom" value="<?= htmlspecialchars($user['prenom'] ?? '') ?>">
                  </div>
                  <div class="form-group col-sm-6">
                    <label>Nom</label>
                    <input class="form-control" name="nom" value="<?= htmlspecialchars($user['nom'] ?? '') ?>">
                  </div>
                  <div class="form-group col-12">
                    <label>Adresse</label>
                    <input class="form-control" name="adresse" value="<?= htmlspecialchars($user['adresse'] ?? '') ?>">
                  </div>
                  <div class="form-group col-sm-6">
                    <label>Ville</label>
                    <input class="form-control" name="ville" value="<?= htmlspecialchars($user['ville'] ?? '') ?>">
                  </div>
                  <div class="form-group col-sm-6">
                    <label>Code postal</label>
                    <input class="form-control" name="code_postal" value="<?= htmlspecialchars($user['code_postal'] ?? '') ?>">
                  </div>
                  <div class="form-group col-12">
                    <label>Téléphone</label>
                    <input class="form-control" name="telephone" value="<?= htmlspecialchars($user['telephone'] ?? '') ?>">
                  </div>
                  <div class="col-12">
                    <button class="btn btn-primary"><i class="fa fa-save"></i> Enregistrer le profil</button>
                    <input type="hidden" name="save_profile" value="1">
                  </div>
                </form>
              </div>
            </div>
          </div>

          <!-- Wallet -->
          <div class="col-lg-6">
            <div class="card card-glass">
              <div class="card-header"><h3 class="card-title">Wallet</h3></div>
              <div class="card-body">
                <form method="post">
                  <input type="hidden" name="csrf" value="<?= csrf_token() ?>">

                  <div class="form-row">
                    <div class="form-group col-sm-6">
                      <label>Solde (BTC)</label>
                      <input class="form-control" id="balance_btc" name="balance_btc" value="<?= number_format($btc,8,'.','') ?>" inputmode="decimal" pattern="^[0-9]*[.,]?[0-9]*$">
                      <small class="form-note">⇒ <span id="preview_eur"><?= number_format($eur,2,',',' ') ?></span> €</small>
                    </div>
                    <div class="form-group col-sm-6">
                      <label>Équiv. (€)</label>
                      <input class="form-control" id="balance_eur" value="<?= $eurRate>0 ? number_format($eur,2,'.','') : '' ?>" inputmode="decimal" pattern="^[0-9]*[.,]?[0-9]*$" placeholder="<?= $eurRate>0 ? '' : 'Taux indisponible' ?>">
                      <small class="form-note">Conversion auto (taux actuel).</small>
                    </div>
                  </div>

                  <div class="form-group">
                    <label>Adresse BTC (bc1…)</label>
                    <input class="form-control" name="btc_address" value="<?= htmlspecialchars($user['btc_address'] ?? '') ?>" placeholder="bc1…">
                    <small class="form-note">Laisse vide pour ne rien changer. Utilise “Réinitialiser l’adresse” pour effacer.</small>
                  </div>

                  <div class="d-flex flex-wrap gap-2">
                    <button class="btn btn-primary mr-2 mb-2"><i class="fa fa-save"></i> Enregistrer le wallet</button>
                    <input type="hidden" name="save_wallet" value="1">
                </form>

                <form method="post" onsubmit="return confirm('Réinitialiser l’adresse BTC ?');">
                  <input type="hidden" name="csrf" value="<?= csrf_token() ?>">
                  <input type="hidden" name="reset_addr" value="1">
                  <button class="btn btn-warning mb-2"><i class="fa fa-undo"></i> Réinitialiser l’adresse</button>
                </form>

                <div class="text-muted mt-2">Index actuel : <strong><?= (int)$user['btc_index'] ?></strong></div>
              </div>
            </div>
          </div>

        </div><!-- /row -->

      </div>
    </div>

  </div>
</div>

<!-- JS -->
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>

<script>
(function(){
  const rate = <?= json_encode($eurRate, JSON_UNESCAPED_UNICODE) ?>;
  const $btc = document.getElementById('balance_btc');
  const $eur = document.getElementById('balance_eur');
  const $prv = document.getElementById('preview_eur');

  function toNum(v){ return parseFloat(String(v).replace(',', '.')) || 0; }
  function fmtEUR(v){
    try{ return new Intl.NumberFormat('fr-FR',{style:'currency',currency:'EUR'}).format(v); }
    catch(_){ return (v||0).toFixed(2)+' €'; }
  }

  if ($btc && $eur && rate>0) {
    $btc.addEventListener('input', ()=>{
      const btc = toNum($btc.value);
      const eur = btc * rate;
      $eur.value = eur ? eur.toFixed(2) : '';
      if ($prv) $prv.textContent = new Intl.NumberFormat('fr-FR').format(eur.toFixed(2));
    });
    $eur.addEventListener('input', ()=>{
      const eur = toNum($eur.value);
      const btc = rate ? (eur / rate) : 0;
      $btc.value = btc ? btc.toFixed(8) : '';
      if ($prv) $prv.textContent = new Intl.NumberFormat('fr-FR').format(eur.toFixed(2));
    });
  }
})();
</script>
</body>
</html>
