<?php
// edit_product.php — édition produit (basé sur ajouter_produit.php)
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

function e($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$success = "";
$error   = "";

// --- Récupération produit ---
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($id <= 0) {
  header("Location: produits.php");
  exit;
}

// Produit
$st = $pdo->prepare("SELECT id, nom, description, image, farm, categorie FROM produits WHERE id = ?");
$st->execute([$id]);
$produit = $st->fetch(PDO::FETCH_ASSOC);
if (!$produit) {
  $error = "❌ Produit introuvable (#$id).";
}

// Prix existants
$prices = [];
if ($produit) {
  $sp = $pdo->prepare("SELECT quantite, prix FROM produit_prix WHERE produit_id = ? ORDER BY quantite ASC");
  $sp->execute([$id]);
  $prices = $sp->fetchAll(PDO::FETCH_ASSOC);
}

// --- Soumission ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $produit) {
    $nom        = trim($_POST['nom'] ?? '');
    $description= trim($_POST['description'] ?? '');
    $farm       = trim($_POST['farm'] ?? '');
    $categorie  = trim($_POST['categorie'] ?? '');

    if ($nom === '' || $farm === '' || $categorie === '') {
      $error = "❌ Champs requis manquants (nom, farm, catégorie).";
    }

    // Image principale : optionnelle (si upload => remplace)
    $image_path = $produit['image']; // garder l'actuelle par défaut
    if (!$error && !empty($_FILES['image_principale']['name'])) {
        $target_dir    = '../uploads/';
        if (!is_dir($target_dir)) mkdir($target_dir);
        $filename      = time() . '_main_' . basename($_FILES['image_principale']['name']);
        $full_path     = $target_dir . $filename;
        $relative_path = 'uploads/' . $filename;

        if (move_uploaded_file($_FILES['image_principale']['tmp_name'], $full_path)) {
            $image_path = $relative_path;

            // (Option) supprimer l’ancienne image locale si c’était un chemin relatif
            $old = trim((string)$produit['image']);
            if ($old && !preg_match('~^(?:https?:)?/|^data:~i', $old)) {
              $root = realpath(dirname(__DIR__));
              if ($root) {
                $oldFull = $root . DIRECTORY_SEPARATOR . ltrim($old, '/\\');
                $oldReal = realpath($oldFull);
                if ($oldReal !== false && strpos($oldReal, $root) === 0 && is_file($oldReal)) {
                  @unlink($oldReal);
                }
              }
            }
        } else {
            $error = "❌ Upload de la nouvelle image principale échoué.";
        }
    }

    // Si pas d'erreur -> update produit
    if (!$error) {
        $stU = $pdo->prepare("UPDATE produits SET nom = ?, description = ?, image = ?, farm = ?, categorie = ? WHERE id = ?");
        $ok  = $stU->execute([$nom, $description, $image_path, $farm, $categorie, $id]);

        if (!$ok) {
          $error = "❌ Erreur lors de la mise à jour du produit.";
        } else {
          // Médias (ajouts optionnels)
          if (!empty($_FILES['medias']['name'][0])) {
            $total = count($_FILES['medias']['name']);
            for ($i=0; $i<$total; $i++){
              $tmp  = $_FILES['medias']['tmp_name'][$i];
              $name = time() . '_' . basename($_FILES['medias']['name'][$i]);

              $dir   = '../uploads/';
              if (!is_dir($dir)) mkdir($dir);
              $path  = $dir . $name;
              $rel   = 'uploads/' . $name;
              $ext   = strtolower(pathinfo($name, PATHINFO_EXTENSION));
              $type  = in_array($ext, ['mp4','webm','mov']) ? 'video' : 'image';

              if (is_uploaded_file($tmp) && move_uploaded_file($tmp, $path)) {
                $pdo->prepare("INSERT INTO produit_medias (produit_id, media_path, type) VALUES (?, ?, ?)")
                    ->execute([$id, $rel, $type]);
              }
            }
          }

          // Prix : on remplace l’ensemble par les lignes envoyées
          $pdo->prepare("DELETE FROM produit_prix WHERE produit_id = ?")->execute([$id]);
          if (!empty($_POST['quantite']) && !empty($_POST['prix'])) {
            foreach ($_POST['quantite'] as $idx => $qte) {
              $qte = trim((string)$qte);
              $prx = $_POST['prix'][$idx] ?? '';
              if ($qte === '' || $prx === '') continue;
              // cast sécurisé
              $prx = str_replace(',', '.', (string)$prx);
              $prx = (float)$prx;
              $pdo->prepare("INSERT INTO produit_prix (produit_id, quantite, prix) VALUES (?, ?, ?)")
                  ->execute([$id, $qte, $prx]);
            }
          }

          $success = "✅ Produit mis à jour avec succès.";
          // rafraîchir $produit et $prices pour l’affichage
          $st->execute([$id]);
          $produit = $st->fetch(PDO::FETCH_ASSOC);
          $sp->execute([$id]);
          $prices  = $sp->fetchAll(PDO::FETCH_ASSOC);
        }
    }
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Modifier un produit</title>
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
  <link rel="stylesheet" href="assets/responsive-fixes.css">
</head>
<body class="hold-transition sidebar-mini layout-fixed">
<div class="wrapper">

<nav class="main-header navbar navbar-expand navbar-white navbar-light">
  <ul class="navbar-nav">
    <li class="nav-item">
      <a class="nav-link" data-widget="pushmenu" href="#" role="button">☰</a>
    </li>
    <li class="nav-item d-sm-none d-block">
      <span class="navbar-text">MeltLabz Admin</span>
    </li>
  </ul>
</nav>

<?php include 'sidebar.php'; ?>

<div class="content-wrapper">
  <div class="content-header">
    <div class="container-fluid">
      <h1 class="m-0">✏️ Modifier un produit</h1>
    </div>
  </div>

  <div class="content">
    <div class="container-fluid">

      <?php if ($error): ?>
        <div class="alert alert-danger"><?= e($error) ?></div>
      <?php elseif ($success): ?>
        <div class="alert alert-success"><?= e($success) ?></div>
      <?php endif; ?>

      <?php if ($produit): ?>
      <div class="card card-light p-4">
        <form method="post" enctype="multipart/form-data">
          <div class="row g-3">

            <div class="col-md-6">
              <label class="form-label">Nom du produit</label>
              <input type="text" name="nom" class="form-control" value="<?= e($produit['nom']) ?>" required>
            </div>

            <div class="col-md-6">
              <label class="form-label">Farm</label>
              <input type="text" name="farm" class="form-control" value="<?= e($produit['farm']) ?>" required>
            </div>

            <div class="col-md-6">
              <label class="form-label">Catégorie</label>
              <select name="categorie" class="form-select form-control" required>
                <?php
                  $opts = ['piatella','wpff','la mousse','iceolator','accessoires'];
                  $cur  = (string)$produit['categorie'];
                  echo '<option value="">-- Choisir --</option>';
                  foreach($opts as $o){
                    $sel = (strcasecmp($cur,$o)===0) ? 'selected' : '';
                    echo '<option value="'.e($o).'" '.$sel.'>'.e($o).'</option>';
                  }
                ?>
              </select>
            </div>

            <div class="col-md-6">
              <label class="form-label">Médias (images/vidéos) — ajouter</label>
              <input type="file" name="medias[]" class="form-control" accept="image/*,video/*" multiple>
              <small class="form-text text-muted">Laisse vide si tu n’ajoutes rien. Les médias existants sont conservés.</small>
            </div>

            <div class="col-md-12">
              <label class="form-label d-block">Image principale (actuelle)</label>
              <?php
                $src = trim((string)($produit['image'] ?? ''));
                if ($src !== '' && !preg_match('~^(?:https?:)?/|^data:~i', $src)) $src = '../'.$src;
              ?>
              <?php if (!empty($produit['image'])): ?>
                <img src="<?= e($src) ?>" alt="" style="height:80px;border-radius:8px;border:1px solid #e5e8ef">
              <?php else: ?>
                <span class="text-muted">—</span>
              <?php endif; ?>
              <div class="mt-2">
                <label class="form-label">Remplacer l’image principale</label>
                <input type="file" name="image_principale" class="form-control" accept="image/*">
                <small class="form-text text-muted">Laisse vide pour conserver l’image actuelle.</small>
              </div>
            </div>

            <div class="col-12">
              <label class="form-label">Description</label>
              <textarea name="description" class="form-control" rows="4" required><?= e($produit['description']) ?></textarea>
            </div>

            <div class="col-12">
              <label class="form-label">Quantité & Prix</label>
              <div id="prix-container">
                <?php if ($prices): foreach ($prices as $p): ?>
                  <div class="row g-2 mb-2">
                    <div class="col-5">
                      <input type="text" name="quantite[]" class="form-control" value="<?= e($p['quantite']) ?>" placeholder="Ex: 5g">
                    </div>
                    <div class="col-5">
                      <input type="number" name="prix[]" step="0.01" class="form-control" value="<?= e(number_format((float)$p['prix'],2,'.','')) ?>" placeholder="Prix en €">
                    </div>
                    <div class="col-2">
                      <button type="button" class="btn btn-outline-danger w-100 remove-row">🗑️</button>
                    </div>
                  </div>
                <?php endforeach; else: ?>
                  <div class="row g-2 mb-2">
                    <div class="col-5">
                      <input type="text" name="quantite[]" class="form-control" placeholder="Ex: 5g">
                    </div>
                    <div class="col-5">
                      <input type="number" name="prix[]" step="0.01" class="form-control" placeholder="Prix en €">
                    </div>
                    <div class="col-2">
                      <button type="button" class="btn btn-outline-danger w-100 remove-row">🗑️</button>
                    </div>
                  </div>
                <?php endif; ?>
              </div>
              <button type="button" id="add-prix" class="btn btn-sm btn-outline-secondary mt-2">➕ Ajouter une ligne</button>
            </div>

            <div class="col-12 text-end">
              <button type="submit" class="btn btn-dark px-4">💾 Enregistrer</button>
              <a href="produits.php" class="btn btn-outline-secondary">↩ Retour</a>
            </div>
          </div>
        </form>
      </div>
      <?php endif; ?>

      <footer class="text-center mt-4 text-muted small">
        &copy; <?= date('Y') ?> MeltLabz
      </footer>

    </div>
  </div>
</div>
</div>

<script>
  // Ajouter / supprimer lignes prix
  document.getElementById('add-prix')?.addEventListener('click', function () {
    const row = document.createElement('div');
    row.className = "row g-2 mb-2";
    row.innerHTML = `
      <div class="col-5">
        <input type="text" name="quantite[]" class="form-control" placeholder="Ex: 5g">
      </div>
      <div class="col-5">
        <input type="number" name="prix[]" step="0.01" class="form-control" placeholder="Prix en €">
      </div>
      <div class="col-2">
        <button type="button" class="btn btn-outline-danger w-100 remove-row">🗑️</button>
      </div>`;
    document.getElementById('prix-container').appendChild(row);
  });

  document.addEventListener('click', function (e) {
    if (e.target.classList.contains('remove-row')) {
      e.target.closest('.row').remove();
    }
  });
</script>
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>
</body>
</html>
