<?php
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
  header('Location: login.php');
  exit();
}

/**
 * Récupère le taux BTC→EUR (cache 60s). On essaie d’abord /api/rates.php,
 * sinon fallback CoinGecko. Si tout échoue, on met 0 (et on affichera —).
 */
function btc_eur_rate(): float {
  $cache = __DIR__ . '/.btc_eur.cache.json';
  $now = time();

  if (is_file($cache) && ($now - filemtime($cache) < 60)) {
    $j = json_decode(@file_get_contents($cache), true);
    if (isset($j['eur']) && $j['eur'] > 0) return (float)$j['eur'];
  }

  $eur = 0.0;
  // 1) endpoint interne
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
  $local = @file_get_contents($scheme.'://'.$host.'/api/rates.php');
  if ($local) {
    $js = json_decode($local, true);
    if (!empty($js['ok']) && !empty($js['btc_eur'])) $eur = (float)$js['btc_eur'];
  }
  // 2) fallback externe
  if ($eur <= 0) {
    $co = @file_get_contents('https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=eur');
    if ($co) {
      $js = json_decode($co, true);
      $eur = (float)($js['bitcoin']['eur'] ?? 0);
    }
  }
  if ($eur > 0) @file_put_contents($cache, json_encode(['eur'=>$eur]));
  return max(0.0, $eur);
}
function sats_to_btc($sats): float { return ((int)$sats)/1e8; }

$eurRate = btc_eur_rate();

// Récupération des utilisateurs (+ ville en plus)
$stmt = $pdo->query("
  SELECT tu.*, a.nom, a.prenom, a.adresse, a.code_postal, a.telephone, a.ville
  FROM telegram_users tu
  LEFT JOIN adresses a ON tu.telegram_id = a.telegram_id
  ORDER BY tu.created_at DESC
");
$clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Clients Telegram</title>

  <!-- Bootstrap 4 + AdminLTE -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">

  <!-- DataTables (version Bootstrap 4 pour matcher ton BS 4.6) -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap4.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
  <link rel="stylesheet" href="assets/responsive-fixes.css">

  <style>
    /* Enveloppe responsive propre pour le tableau */
    .table-responsive-wrap { overflow-x:auto; }
    /* Badge/étiquette ID plus lisible */
    .badge-id { background:#eef0f4; color:#333; border:1px solid #dfe3ea; font-weight:600; }
    /* Colonne actions compacte sur mobile */
    @media (max-width: 576px){
      td.actions > a { display:block; margin-bottom:6px; }
    }
  </style>
</head>
<body class="hold-transition sidebar-mini layout-fixed">
<div class="wrapper">

  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item">
        <a class="nav-link" data-widget="pushmenu" href="#" role="button">☰</a>
      </li>
      <li class="nav-item d-sm-none d-block">
        <span class="navbar-text">MeltLabz Admin</span>
      </li>
    </ul>
  </nav>

  <?php include 'sidebar.php'; ?>

  <div class="content-wrapper">
    <div class="content-header">
      <div class="container-fluid">
        <?php if (!empty($_SESSION['flash_message'])): ?>
          <div class="alert alert-info alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($_SESSION['flash_message']) ?>
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
              <span aria-hidden="true">&times;</span>
            </button>
          </div>
          <?php unset($_SESSION['flash_message']); ?>
        <?php endif; ?>
        <h1 class="m-0">👥 Clients Telegram</h1>
        <p class="text-muted mb-0">
          Taux BTC→€ actuel :
          <strong><?= $eurRate>0 ? number_format($eurRate, 2, ',', ' ') . ' € / BTC' : '—' ?></strong>
        </p>
      </div>
    </div>

    <div class="content">
      <div class="container-fluid">
        <div class="card">
          <div class="card-body table-responsive-wrap">

            <table id="clientsTable" class="table table-bordered table-hover">
              <thead>
                <tr>
                  <th>Avatar</th>
                  <th>@Telegram</th>
                  <th>Nom</th>
                  <th>Adresse</th>
                  <th>Ville</th>
                  <th>Téléphone</th>
                  <th>Solde €</th>
                  <th class="text-muted">BTC</th>
                  <th>Actions</th>
                </tr>
              </thead>
              <tbody>
              <?php foreach ($clients as $client):
                $btc = sats_to_btc($client['balance_sats'] ?? 0);
                $eur = $eurRate>0 ? $btc * $eurRate : 0;
                $username = $client['username'] ? '@'.htmlspecialchars($client['username']) : '—';
                $id = (int)$client['telegram_id'];
              ?>
                <tr>
                  <td>
                    <img src="<?= htmlspecialchars($client['photo_url']) ?>" alt="avatar" width="40" class="rounded-circle">
                  </td>

                  <td>
                    <?= $username ?><br>
                    <span class="badge badge-id">ID: <?= $id ?></span>
                  </td>

                  <td><?= htmlspecialchars(trim(($client['prenom'] ?? '').' '.($client['nom'] ?? ''))) ?></td>

                  <td><?= htmlspecialchars(trim(($client['adresse'] ?? '').' '.($client['code_postal'] ?? ''))) ?></td>

                  <td><?= htmlspecialchars($client['ville'] ?? '—') ?></td>

                  <td><?= htmlspecialchars($client['telephone'] ?? '—') ?></td>

                  <td><strong><?= $eurRate>0 ? number_format($eur, 2, ',', ' ') : '—' ?> €</strong></td>

                  <td class="text-muted"><?= number_format($btc, 8, '.', '') ?> BTC</td>

                  <td class="actions" style="min-width:210px">
                    <!-- Modifier -->
                    <a href="edit_user.php?id=<?= $id ?>" class="btn btn-primary btn-sm">
                      <i class="fa fa-pen"></i> Modifier
                    </a>

                    <!-- Bannir / Débannir (on garde exactement ta logique/URLs) -->
                    <?php if (!empty($client['is_banned'])): ?>
                      <a href="toggle_ban.php?id=<?= $id ?>&action=unban" class="btn btn-success btn-sm">
                        <i class="fa fa-unlock"></i> Débannir
                      </a>
                    <?php else: ?>
                      <a href="toggle_ban.php?id=<?= $id ?>&action=ban" class="btn btn-danger btn-sm" onclick="return confirm('Bannir cet utilisateur ?');">
                        <i class="fa fa-ban"></i> Bannir
                      </a>
                    <?php endif; ?>
                  </td>
                </tr>
              <?php endforeach; ?>
              </tbody>
            </table>

          </div>
        </div>
      </div>
    </div>

  </div>
</div>

<!-- jQuery puis Bootstrap, puis AdminLTE -->
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>

<!-- DataTables (JS bootstrap4 pour matcher ton thème) -->
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap4.min.js"></script>

<script>
  $(function () {
    $('#clientsTable').DataTable({
      language: { url: '//cdn.datatables.net/plug-ins/1.13.6/i18n/fr-FR.json' },
      pageLength: 10,
      lengthMenu: [ [10, 20, 50, 100], [10, 20, 50, 100] ],
      order: [[ 1, 'desc' ]], // tri par @Telegram (ID affiché en badge)
      responsive: false // on gère nous-mêmes via .table-responsive-wrap
    });
  });
</script>
</body>
</html>
