<?php
// bot_telegram.php — Gestion du bot Telegram (webhook + boutons)
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

/* ==========================
   Helpers
   ========================== */
function e($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}

/**
 * Récupère (ou crée) le bot principal (id=1) avec les valeurs de ton script Python
 */
function tg_get_bot_or_create(PDO $pdo): array {
    $st = $pdo->prepare("SELECT * FROM telegram_bots WHERE id = 1 LIMIT 1");
    $st->execute();
    $bot = $st->fetch(PDO::FETCH_ASSOC);

    if (!$bot) {
        $sql = "INSERT INTO telegram_bots
                (id, title, token, bot_username, group_chat_id, webhook_url, gif_url, image_enabled, chat_url, contact_url, is_active)
                VALUES
                (1,'Senzu Asia Bot',
                 '8058105739:AAGYQ2goMQqS1KOTaHQ9e6zTIfcTDJv1MiA',
                 'MeltLabzBot',
                 -1001759914839,
                 'https://meltlabz.us/bot/webhook.php',
                 'https://meltlabz.us/assets/loaders.gif',
                 1,
                 'https://t.me/MeltLabz',
                 'https://t.me/MeltLabzUS',
                 0)";
        $pdo->exec($sql);
        $st->execute();
        $bot = $st->fetch(PDO::FETCH_ASSOC) ?: [];
    }

    return $bot;
}

/**
 * Appel API Telegram simple (setWebhook / deleteWebhook)
 */
function tg_api_call(string $token, string $method, array $params = []): array {
    if (trim($token) === '') {
        return ['ok' => false, 'description' => 'Token manquant'];
    }

    $url = 'https://api.telegram.org/bot' . rawurlencode($token) . '/' . $method;
    $opts = [
        'http' => [
            'method'  => 'POST',
            'header'  => "Content-Type: application/x-www-form-urlencoded\r\n",
            'content' => http_build_query($params),
            'timeout' => 10,
        ],
    ];
    $ctx  = stream_context_create($opts);
    $resp = @file_get_contents($url, false, $ctx);
    if ($resp === false) {
        return ['ok' => false, 'description' => 'Erreur HTTP'];
    }

    $data = json_decode($resp, true);
    if (!is_array($data)) {
        return ['ok' => false, 'description' => 'Réponse JSON invalide'];
    }
    return $data;
}

/**
 * Récupère les boutons, groupés par location/lang/row_index
 *
 * @return array<string,array<string,array<int,array<int,array>>>>  location -> lang -> row -> [buttons...]
 */
function tg_fetch_buttons_tree(PDO $pdo, int $botId): array {
    $st = $pdo->prepare("
        SELECT *
        FROM telegram_bot_buttons
        WHERE bot_id = :bot_id
        ORDER BY location, lang, row_index, col_index, id
    ");
    $st->execute([':bot_id' => $botId]);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];

    $tree = [];
    foreach ($rows as $btn) {
        $loc   = $btn['location'];
        $lang  = $btn['lang'];
        $rowIx = (int)$btn['row_index'];

        if (!isset($tree[$loc])) {
            $tree[$loc] = [];
        }
        if (!isset($tree[$loc][$lang])) {
            $tree[$loc][$lang] = [];
        }
        if (!isset($tree[$loc][$lang][$rowIx])) {
            $tree[$loc][$lang][$rowIx] = [];
        }
        $tree[$loc][$lang][$rowIx][] = $btn;
    }
    return $tree;
}

/**
 * Calcul du prochain col_index pour une ligne donnée
 */
function tg_next_col_index(PDO $pdo, int $botId, string $location, string $lang, int $rowIndex): int {
    $st = $pdo->prepare("
        SELECT COALESCE(MAX(col_index) + 1, 0) AS next_col
        FROM telegram_bot_buttons
        WHERE bot_id = :bot_id AND location = :loc AND lang = :lang AND row_index = :row_ix
    ");
    $st->execute([
        ':bot_id' => $botId,
        ':loc'    => $location,
        ':lang'   => $lang,
        ':row_ix' => $rowIndex,
    ]);
    $next = $st->fetchColumn();
    return (int)($next ?? 0);
}

/* ==========================
   Actions AJAX (POST)
   ========================== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json; charset=utf-8');

    $action = $_POST['action'] ?? '';
    $res = ['ok' => false, 'message' => 'Action inconnue.'];

    try {
        /** @var PDO $pdo */
        $bot = tg_get_bot_or_create($pdo);
        $botId = (int)$bot['id'];

        if ($action === 'save_bot') {
            $token       = trim($_POST['token'] ?? '');
            $botUser     = trim($_POST['bot_username'] ?? '');
            $groupIdRaw  = trim($_POST['group_chat_id'] ?? '');
            $webhookUrl  = trim($_POST['webhook_url'] ?? '');
            $gifUrl      = trim($_POST['gif_url'] ?? '');
            $chatUrl     = trim($_POST['chat_url'] ?? '');
            $contactUrl  = trim($_POST['contact_url'] ?? '');
            $imageEnabled = isset($_POST['image_enabled']) ? 1 : 0;

            $groupId = ($groupIdRaw !== '' ? $groupIdRaw : null);

            $st = $pdo->prepare("
                UPDATE telegram_bots
                SET token = :token,
                    bot_username = :bot_username,
                    group_chat_id = :group_chat_id,
                    webhook_url = :webhook_url,
                    gif_url = :gif_url,
                    chat_url = :chat_url,
                    contact_url = :contact_url,
                    image_enabled = :image_enabled
                WHERE id = :id
            ");
            $st->execute([
                ':token'        => $token,
                ':bot_username' => $botUser,
                ':group_chat_id'=> $groupId,
                ':webhook_url'  => $webhookUrl,
                ':gif_url'      => $gifUrl,
                ':chat_url'     => $chatUrl,
                ':contact_url'  => $contactUrl,
                ':image_enabled'=> $imageEnabled,
                ':id'           => $botId,
            ]);

            $res['ok'] = true;
            $res['message'] = 'Paramètres bot sauvegardés.';

        } elseif ($action === 'start_webhook') {
            $token      = trim((string)$bot['token'] ?? '');
            $webhookUrl = trim((string)$bot['webhook_url'] ?? '');

            if ($token === '' || $webhookUrl === '') {
                $res['message'] = 'Token ou Webhook URL manquant.';
            } else {
                $api = tg_api_call($token, 'setWebhook', ['url' => $webhookUrl]);
                if (!empty($api['ok'])) {
                    $pdo->prepare("UPDATE telegram_bots SET is_active = 1 WHERE id = ?")->execute([$botId]);
                    $res['ok'] = true;
                    $res['message'] = 'Webhook démarré avec succès ✅';
                } else {
                    $res['message'] = 'Erreur setWebhook : ' . ($api['description'] ?? 'inconnue');
                }
            }

        } elseif ($action === 'stop_webhook') {
            $token = trim((string)$bot['token'] ?? '');

            if ($token === '') {
                $res['message'] = 'Token manquant.';
            } else {
                $api = tg_api_call($token, 'deleteWebhook', []);
                if (!empty($api['ok'])) {
                    $pdo->prepare("UPDATE telegram_bots SET is_active = 0 WHERE id = ?")->execute([$botId]);
                    $res['ok'] = true;
                    $res['message'] = 'Webhook stoppé ⛔';
                } else {
                    $res['message'] = 'Erreur deleteWebhook : ' . ($api['description'] ?? 'inconnue');
                }
            }

        } elseif ($action === 'add_button') {
            $location   = $_POST['location'] ?? 'dm';  // 'dm' ou 'group'
            $lang       = $_POST['lang'] ?? 'fr';      // 'fr','en','es'
            $label      = trim($_POST['label'] ?? '');
            $rowIndex   = (int)($_POST['row_index'] ?? 0);
            $actionType = $_POST['action_type'] ?? 'url';
            $target     = trim($_POST['target'] ?? '');

            if ($label === '' || $target === '') {
                $res['message'] = 'Label et cible sont obligatoires.';
                echo json_encode($res);
                exit;
            }

            if (!in_array($location, ['dm','group'], true)) {
                $location = 'dm';
            }
            if (!in_array($lang, ['fr','en','es'], true)) {
                $lang = 'en';
            }
            if (!in_array($actionType, ['url','webapp','callback'], true)) {
                $actionType = 'url';
            }

            $colIndex = tg_next_col_index($pdo, $botId, $location, $lang, $rowIndex);

            $url = $webappUrl = $cb = null;
            if ($actionType === 'url') {
                $url = $target;
            } elseif ($actionType === 'webapp') {
                $webappUrl = $target;
            } else {
                $cb = $target;
            }

            $st = $pdo->prepare("
                INSERT INTO telegram_bot_buttons
                  (bot_id, location, lang, row_index, col_index, label, action_type, url, webapp_url, callback_data, is_active)
                VALUES
                  (:bot_id, :loc, :lang, :row_ix, :col_ix, :label, :action_type, :url, :webapp, :cb, 1)
            ");
            $st->execute([
                ':bot_id'      => $botId,
                ':loc'         => $location,
                ':lang'        => $lang,
                ':row_ix'      => $rowIndex,
                ':col_ix'      => $colIndex,
                ':label'       => $label,
                ':action_type' => $actionType,
                ':url'         => $url,
                ':webapp'      => $webappUrl,
                ':cb'          => $cb,
            ]);

            $res['ok'] = true;
            $res['message'] = 'Bouton ajouté.';

        } elseif ($action === 'update_button') {
            $id         = (int)($_POST['id'] ?? 0);
            $label      = trim($_POST['label'] ?? '');
            $rowIndex   = (int)($_POST['row_index'] ?? 0);
            $colIndex   = (int)($_POST['col_index'] ?? 0);
            $actionType = $_POST['action_type'] ?? 'url';
            $target     = trim($_POST['target'] ?? '');

            if ($id <= 0) {
                $res['message'] = 'ID bouton invalide.';
                echo json_encode($res);
                exit;
            }
            if ($label === '' || $target === '') {
                $res['message'] = 'Label et cible sont obligatoires.';
                echo json_encode($res);
                exit;
            }

            if (!in_array($actionType, ['url','webapp','callback'], true)) {
                $actionType = 'url';
            }

            $url = $webappUrl = $cb = null;
            if ($actionType === 'url') {
                $url = $target;
            } elseif ($actionType === 'webapp') {
                $webappUrl = $target;
            } else {
                $cb = $target;
            }

            $st = $pdo->prepare("
                UPDATE telegram_bot_buttons
                SET label = :label,
                    row_index = :row_ix,
                    col_index = :col_ix,
                    action_type = :action_type,
                    url = :url,
                    webapp_url = :webapp,
                    callback_data = :cb
                WHERE id = :id AND bot_id = :bot_id
            ");
            $st->execute([
                ':label'       => $label,
                ':row_ix'      => $rowIndex,
                ':col_ix'      => $colIndex,
                ':action_type' => $actionType,
                ':url'         => $url,
                ':webapp'      => $webappUrl,
                ':cb'          => $cb,
                ':id'          => $id,
                ':bot_id'      => $botId,
            ]);

            $res['ok'] = true;
            $res['message'] = 'Bouton mis à jour.';

        } elseif ($action === 'delete_button') {
            $id = (int)($_POST['id'] ?? 0);
            if ($id <= 0) {
                $res['message'] = 'ID bouton invalide.';
                echo json_encode($res);
                exit;
            }

            $st = $pdo->prepare("DELETE FROM telegram_bot_buttons WHERE id = :id AND bot_id = :bot_id");
            $st->execute([
                ':id'     => $id,
                ':bot_id' => $botId,
            ]);

            $res['ok'] = true;
            $res['message'] = 'Bouton supprimé.';
        }

    } catch (Throwable $e) {
        $res['ok'] = false;
        $res['message'] = 'Erreur serveur : ' . $e->getMessage();
    }

    echo json_encode($res);
    exit;
}

/* ==========================
   GET : affichage
   ========================== */

/** @var PDO $pdo */
$bot    = tg_get_bot_or_create($pdo);
$botId  = (int)$bot['id'];
$tree   = tg_fetch_buttons_tree($pdo, $botId);

$langs = [
    'fr' => 'Français',
    'en' => 'English',
    'es' => 'Español',
];
$locations = [
    'dm'    => 'Clavier DM (WebApps + Chat/Contact)',
    'group' => 'Clavier Groupe (Welcome + Langue)',
];

$isActive = !empty($bot['is_active']);

?>
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Telegram Bot - MeltLabz</title>

  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
  <link rel="stylesheet" href="assets/responsive-fixes.css">

  <style>
    .tg-status-badge{
      font-size:.75rem;
      padding:.3rem .6rem;
      border-radius:999px;
    }
    .tg-status-badge.active{background:#16a34a;color:#fff;}
    .tg-status-badge.inactive{background:#4b5563;color:#e5e7eb;}

    .tg-accordion .card-header{
      background:#0b1120;
      border-bottom:1px solid rgba(148,163,184,.4);
    }
    .tg-accordion .card{
      background:#020617;
      border:1px solid rgba(148,163,184,.35);
      border-radius:12px;
      overflow:hidden;
    }
    .tg-btn-pill{
      border-radius:999px;
      border:1px solid rgba(148,163,184,.6);
      background:rgba(15,23,42,.9);
      color:#e5e7eb;
      padding:.35rem .75rem;
      font-size:.8rem;
      margin:2px;
      display:inline-flex;
      flex-direction:column;
      align-items:flex-start;
      cursor:pointer;
      transition:background .15s, transform .1s, box-shadow .1s;
    }
    .tg-btn-pill:hover{
      background:#111827;
      transform:translateY(-1px);
      box-shadow:0 8px 20px rgba(0,0,0,.45);
    }
    .tg-btn-pill span.label{
      font-weight:600;
    }
    .tg-btn-pill span.meta{
      font-size:.7rem;
      color:#9ca3af;
    }
    .tg-add-card{
      background:linear-gradient(135deg,#111827,#020617);
      border-radius:12px;
      border:1px dashed rgba(148,163,184,.6);
    }
  </style>
</head>
<body class="hold-transition sidebar-mini">
<div class="wrapper">

  <!-- Navbar -->
  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item">
        <a class="nav-link" data-widget="pushmenu" href="#" role="button">
          <i class="fas fa-bars"></i>
        </a>
      </li>
      <li class="nav-item d-none d-sm-inline-block">
        <a class="nav-link">MeltLabz Admin</a>
      </li>
    </ul>
  </nav>

  <?php include 'sidebar.php'; ?>

  <div class="content-wrapper p-3">
    <div class="container-fluid">

      <div class="d-flex flex-wrap align-items-center mb-3" style="gap:10px">
        <h1 class="h4 m-0 flex-grow-1">Telegram Bot</h1>
        <span class="tg-status-badge <?= $isActive ? 'active' : 'inactive' ?>">
          <?= $isActive ? 'Webhook actif' : 'Webhook stoppé' ?>
        </span>
      </div>

      <div class="row">
        <!-- Colonne gauche : paramètres + start/stop -->
        <div class="col-lg-4 mb-3">
          <div class="card card-dark">
            <div class="card-body">
              <h5 class="card-title mb-3">Paramètres du bot</h5>

              <form id="bot-settings-form">
                <div id="bot-settings-alert" class="alert alert-danger small d-none"></div>
                <input type="hidden" name="action" value="save_bot">

                <div class="form-group">
                  <label>Bot token</label>
                  <input type="text" class="form-control" name="token"
                         value="<?= e($bot['token'] ?? '') ?>">
                  <small class="text-muted">Token donné par BotFather.</small>
                </div>

                <div class="form-group">
                  <label>Bot username (sans @)</label>
                  <input type="text" class="form-control" name="bot_username"
                         value="<?= e($bot['bot_username'] ?? 'MeltLabzBot') ?>">
                </div>

                <div class="form-group">
                  <label>Group chat ID</label>
                  <input type="text" class="form-control" name="group_chat_id"
                         value="<?= e($bot['group_chat_id'] ?? '-1001759914839') ?>">
                  <small class="text-muted">Ex: -1001759914839 (commande /chatid dans ton script).</small>
                </div>

                <div class="form-group">
                  <label>Webhook URL</label>
                  <input type="text" class="form-control" name="webhook_url"
                         value="<?= e($bot['webhook_url'] ?? 'https://meltlabz.us/bot/webhook.php') ?>">
                  <small class="text-muted">URL publique qui reçoit les updates Telegram.</small>
                </div>

                <div class="custom-control custom-switch mb-2">
                  <input type="checkbox" class="custom-control-input" id="image_enabled"
                         name="image_enabled" <?= !empty($bot['image_enabled']) ? 'checked' : '' ?>>
                  <label class="custom-control-label" for="image_enabled">
                    Activer l'image / GIF de welcome
                  </label>
                </div>

                <div class="form-group">
                  <label>GIF / Image URL (GIF_URL)</label>
                  <input type="text" class="form-control" name="gif_url"
                         value="<?= e($bot['gif_url'] ?? 'https://meltlabz.us/assets/loaders.gif') ?>">
                </div>

                <div class="form-group">
                  <label>Chat URL</label>
                  <input type="text" class="form-control" name="chat_url"
                         value="<?= e($bot['chat_url'] ?? 'https://t.me/MeltLabz') ?>">
                </div>

                <div class="form-group">
                  <label>Contact URL</label>
                  <input type="text" class="form-control" name="contact_url"
                         value="<?= e($bot['contact_url'] ?? 'https://t.me/MeltLabzUS') ?>">
                </div>

                <button type="submit" class="btn btn-primary btn-block mb-2">
                  💾 Sauvegarder les paramètres
                </button>
              </form>

              <div class="d-flex" style="gap:8px;">
                <button type="button" class="btn btn-success flex-fill" id="btn-start-webhook">
                  ▶️ Start webhook
                </button>
                <button type="button" class="btn btn-outline-danger flex-fill" id="btn-stop-webhook">
                  ⏹ Stop webhook
                </button>
              </div>
            </div>
          </div>
        </div>

        <!-- Colonne droite : arborescense des boutons -->
        <div class="col-lg-8 mb-3">
          <div class="card card-dark">
            <div class="card-body">
              <h5 class="card-title mb-2">Arborescence des boutons</h5>
              <p class="text-muted small">
                Basé sur ton script Python : <strong>keyboard_dm</strong> & <strong>keyboard_group</strong>, multi-langues (FR / EN / ES),
                mêmes URLs, mêmes labels, mêmes WebApps.
              </p>

              <div class="accordion tg-accordion" id="tgButtonsAccordion">
                <?php foreach ($locations as $locKey => $locLabel): ?>
                  <?php $locId = 'loc_' . $locKey; ?>
                  <div class="card mb-2">
                    <div class="card-header" id="<?= $locId ?>_head">
                      <h2 class="mb-0">
                        <button class="btn btn-link text-left text-light" type="button"
                                data-toggle="collapse"
                                data-target="#<?= $locId ?>_body"
                                aria-expanded="false"
                                aria-controls="<?= $locId ?>_body">
                          <?= e($locLabel) ?>
                        </button>
                      </h2>
                    </div>
                    <div id="<?= $locId ?>_body" class="collapse"
                         aria-labelledby="<?= $locId ?>_head"
                         data-parent="#tgButtonsAccordion">
                      <div class="card-body">

                        <ul class="nav nav-pills mb-3" role="tablist">
                          <?php $first = true; ?>
                          <?php foreach ($langs as $langKey => $langLabel): ?>
                            <?php $tabId = $locId . '_' . $langKey; ?>
                            <li class="nav-item">
                              <a class="nav-link <?= $first ? 'active' : '' ?>"
                                 id="<?= $tabId ?>-tab"
                                 data-toggle="tab"
                                 href="#<?= $tabId ?>"
                                 role="tab">
                                <?= $langKey === 'fr' ? '🇫🇷' : ($langKey === 'en' ? '🇺🇸' : '🇪🇸') ?>
                                <?= e($langLabel) ?>
                              </a>
                            </li>
                            <?php $first = false; ?>
                          <?php endforeach; ?>
                        </ul>

                        <div class="tab-content">
                          <?php $firstTab = true; ?>
                          <?php foreach ($langs as $langKey => $langLabel): ?>
                            <?php
                              $tabId   = $locId . '_' . $langKey;
                              $rowsLang = $tree[$locKey][$langKey] ?? [];
                              ksort($rowsLang);
                            ?>
                            <div class="tab-pane fade <?= $firstTab ? 'show active' : '' ?>"
                                 id="<?= $tabId ?>"
                                 role="tabpanel">
                              <?php if (!$rowsLang): ?>
                                <p class="text-muted small mb-2">Aucun bouton pour cette langue.</p>
                              <?php else: ?>
                                <?php foreach ($rowsLang as $rowIx => $buttons): ?>
                                  <div class="mb-2">
                                    <div class="small text-muted mb-1">
                                      Ligne <?= (int)$rowIx ?> :
                                    </div>
                                    <div>
                                      <?php foreach ($buttons as $btn): ?>
                                        <?php
                                          $id   = (int)$btn['id'];
                                          $lab  = (string)$btn['label'];
                                          $type = (string)$btn['action_type'];
                                          if ($type === 'webapp') {
                                              $target = (string)$btn['webapp_url'];
                                          } elseif ($type === 'callback') {
                                              $target = (string)$btn['callback_data'];
                                          } else {
                                              $target = (string)$btn['url'];
                                          }
                                          $shortTarget = mb_strimwidth($target, 0, 45, '…', 'UTF-8');
                                        ?>
                                        <button type="button"
                                                class="tg-btn-pill tg-btn-edit"
                                                data-id="<?= $id ?>"
                                                data-label="<?= e($lab) ?>"
                                                data-location="<?= e($locKey) ?>"
                                                data-lang="<?= e($langKey) ?>"
                                                data-row="<?= (int)$btn['row_index'] ?>"
                                                data-col="<?= (int)$btn['col_index'] ?>"
                                                data-type="<?= e($type) ?>"
                                                data-target="<?= e($target) ?>">
                                          <span class="label"><?= e($lab) ?></span>
                                          <span class="meta">
                                            <?= strtoupper($type) ?> · <?= e($shortTarget) ?>
                                          </span>
                                        </button>
                                      <?php endforeach; ?>
                                    </div>
                                  </div>
                                <?php endforeach; ?>
                              <?php endif; ?>
                            </div>
                            <?php $firstTab = false; ?>
                          <?php endforeach; ?>
                        </div>

                      </div>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>

              <!-- Bloc ajout rapide -->
              <div class="mt-3 p-3 tg-add-card">
                <h6 class="mb-2">➕ Ajouter un bouton</h6>
                <form id="tg-add-button-form">
                  <input type="hidden" name="action" value="add_button">
                  <div id="tg-add-alert" class="alert alert-danger small d-none"></div>
                  <div class="form-row">
                    <div class="form-group col-md-3">
                      <label>Emplacement</label>
                      <select name="location" class="form-control form-control-sm">
                        <option value="dm">DM</option>
                        <option value="group">Groupe</option>
                      </select>
                    </div>
                    <div class="form-group col-md-3">
                      <label>Langue</label>
                      <select name="lang" class="form-control form-control-sm">
                        <option value="fr">Français</option>
                        <option value="en">English</option>
                        <option value="es">Español</option>
                      </select>
                    </div>
                    <div class="form-group col-md-2">
                      <label>Ligne</label>
                      <input type="number" name="row_index" class="form-control form-control-sm" value="0">
                    </div>
                    <div class="form-group col-md-4">
                      <label>Label</label>
                      <input type="text" name="label" class="form-control form-control-sm" placeholder="Texte du bouton">
                    </div>
                  </div>
                  <div class="form-row">
                    <div class="form-group col-md-3">
                      <label>Type</label>
                      <select name="action_type" class="form-control form-control-sm">
                        <option value="webapp">WebApp</option>
                        <option value="url">URL</option>
                        <option value="callback">Callback</option>
                      </select>
                    </div>
                    <div class="form-group col-md-9">
                      <label>Cible (URL / WebApp / callback_data)</label>
                      <input type="text" name="target" class="form-control form-control-sm"
                             placeholder="https://... ou lang_menu">
                    </div>
                  </div>
                  <div class="text-right">
                    <button type="submit" class="btn btn-sm btn-success">
                      Ajouter le bouton
                    </button>
                  </div>
                </form>
              </div>

            </div>
          </div>
        </div>

      </div> <!-- row -->

    </div>
  </div>

</div>

<!-- Modal édition bouton -->
<div class="modal fade" id="tgButtonModal" tabindex="-1" aria-labelledby="tgButtonModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content bg-dark text-light">
      <div class="modal-header border-0">
        <h5 class="modal-title" id="tgButtonModalLabel">Éditer le bouton</h5>
        <button type="button" class="close text-light" data-dismiss="modal" aria-label="Close">
          <span aria-hidden="true">&times;</span>
        </button>
      </div>
      <form id="tg-edit-button-form">
        <div class="modal-body">
          <div id="tg-edit-alert" class="alert alert-danger small d-none"></div>
          <input type="hidden" name="action" value="update_button">
          <input type="hidden" name="id" id="tg-edit-id">

          <div class="form-row">
            <div class="form-group col-md-4">
              <label>Label</label>
              <input type="text" class="form-control" name="label" id="tg-edit-label">
            </div>
            <div class="form-group col-md-2">
              <label>Ligne</label>
              <input type="number" class="form-control" name="row_index" id="tg-edit-row">
            </div>
            <div class="form-group col-md-2">
              <label>Colonne</label>
              <input type="number" class="form-control" name="col_index" id="tg-edit-col">
            </div>
            <div class="form-group col-md-4">
              <label>Type</label>
              <select class="form-control" name="action_type" id="tg-edit-type">
                <option value="webapp">WebApp</option>
                <option value="url">URL</option>
                <option value="callback">Callback</option>
              </select>
            </div>
          </div>

          <div class="form-group">
            <label>Cible (URL / WebApp / callback_data)</label>
            <input type="text" class="form-control" name="target" id="tg-edit-target">
          </div>

        </div>
        <div class="modal-footer border-0">
          <button type="button" class="btn btn-secondary" data-dismiss="modal">Annuler</button>
          <button type="button" class="btn btn-danger" id="tg-btn-delete">
            🗑 Supprimer
          </button>
          <button type="submit" class="btn btn-primary">💾 Sauvegarder</button>
        </div>
      </form>
    </div>
  </div>
</div>


<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>

<script>
$(function(){

  function ajaxPost(data, onDone){
    $.post('bot_telegram.php', data, function(res){
      if (typeof onDone === 'function') onDone(res);
    }, 'json').fail(function(){
      alert('Erreur réseau ou serveur.');
    });
  }

  // Save bot settings
  $('#bot-settings-form').on('submit', function(e){
    e.preventDefault();
    const $btn = $(this).find('button[type="submit"]');
    const $alert = $('#bot-settings-alert');
    $btn.prop('disabled', true).text('Enregistrement...');
    $alert.addClass('d-none').text('');

    ajaxPost($(this).serialize(), function(res){
      if (res && res.ok) {
        $alert.removeClass('d-none').removeClass('alert-danger')
              .addClass('alert-success').text(res.message || 'OK');
        setTimeout(function(){ location.reload(); }, 800);
      } else {
        $alert.removeClass('d-none').removeClass('alert-success')
              .addClass('alert-danger')
              .text(res && res.message ? res.message : 'Erreur inconnue');
      }
      $btn.prop('disabled', false).text('💾 Sauvegarder les paramètres');
    });
  });

  // Start webhook
  $('#btn-start-webhook').on('click', function(){
    ajaxPost({ action: 'start_webhook' }, function(res){
      alert(res && res.message ? res.message : 'Réponse inconnue.');
      if (res && res.ok) location.reload();
    });
  });

  // Stop webhook
  $('#btn-stop-webhook').on('click', function(){
    ajaxPost({ action: 'stop_webhook' }, function(res){
      alert(res && res.message ? res.message : 'Réponse inconnue.');
      if (res && res.ok) location.reload();
    });
  });

  // Ajouter un bouton
  $('#tg-add-button-form').on('submit', function(e){
    e.preventDefault();
    const $btn = $(this).find('button[type="submit"]');
    const $alert = $('#tg-add-alert');
    $btn.prop('disabled', true).text('Ajout...');
    $alert.addClass('d-none').text('');

    ajaxPost($(this).serialize(), function(res){
      if (res && res.ok) {
        location.reload();
      } else {
        $alert.removeClass('d-none').text(res && res.message ? res.message : 'Erreur.');
      }
      $btn.prop('disabled', false).text('Ajouter le bouton');
    });
  });

  // Ouvrir modale édition
  $(document).on('click', '.tg-btn-edit', function(){
    const $b = $(this);
    $('#tg-edit-id').val($b.data('id'));
    $('#tg-edit-label').val($b.data('label'));
    $('#tg-edit-row').val($b.data('row'));
    $('#tg-edit-col').val($b.data('col'));
    $('#tg-edit-type').val($b.data('type'));
    $('#tg-edit-target').val($b.data('target'));
    $('#tg-edit-alert').addClass('d-none').text('');
    $('#tgButtonModalLabel').text('Éditer le bouton #' + $b.data('id'));
    $('#tgButtonModal').modal('show');
  });

  // Sauvegarder bouton
  $('#tg-edit-button-form').on('submit', function(e){
    e.preventDefault();
    const $btn = $(this).find('button[type="submit"]');
    const $alert = $('#tg-edit-alert');
    $btn.prop('disabled', true).text('Enregistrement...');
    $alert.addClass('d-none').text('');

    ajaxPost($(this).serialize(), function(res){
      if (res && res.ok) {
        location.reload();
      } else {
        $alert.removeClass('d-none').text(res && res.message ? res.message : 'Erreur.');
      }
      $btn.prop('disabled', false).text('💾 Sauvegarder');
    });
  });

  // Delete bouton depuis la modale
  $('#tg-btn-delete').on('click', function(){
    const id = $('#tg-edit-id').val();
    if (!id) return;
    if (!confirm('Supprimer le bouton #' + id + ' ?')) return;

    ajaxPost({ action:'delete_button', id:id }, function(res){
      alert(res && res.message ? res.message : 'Réponse inconnue.');
      if (res && res.ok) location.reload();
    });
  });

});
</script>
</body>
</html>