<?php
// Central guard for admin pages: enables "remember me" via cookie.
if (session_status() !== PHP_SESSION_ACTIVE) { session_start(); }

// If already logged in, nothing to do
if (isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
    return;
}

$cookie_name = 'admin_remember';
$store_path  = __DIR__ . '/remember_tokens.json';

// Helper: safely read the token store
function _remember_store_read($store_path){
    if (!file_exists($store_path)) return [];
    $json = @file_get_contents($store_path);
    if ($json === false) return [];
    $data = json_decode($json, true);
    return is_array($data) ? $data : [];
}

// Helper: prune expired tokens and write back
function _remember_store_write($store_path, $data){
    // prune
    $now = time();
    $data = array_values(array_filter($data, function($item) use ($now){
        return isset($item['expires']) && $item['expires'] > $now && isset($item['token']);
    }));
    @file_put_contents($store_path, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
}

if (!empty($_COOKIE[$cookie_name])) {
    $raw = $_COOKIE[$cookie_name];
    // Compare hashed token to avoid exposing raw tokens in store
    $hash = hash('sha256', $raw);

    $store = _remember_store_read($store_path);
    $now = time();
    $match = null;
    foreach ($store as $item){
        if (!isset($item['token']) || !isset($item['expires'])) continue;
        if (hash_equals($item['token'], $hash) && $item['expires'] > $now){
            $match = $item;
            break;
        }
    }
    if ($match){
        // Restore session
        $_SESSION['admin_logged_in'] = true;
        if (isset($match['username'])) $_SESSION['admin_username'] = $match['username'];

        // Optional: sliding expiration extend by 7 days if near expiry
        if ($match['expires'] - $now < 7*24*3600){
            $newExp = $now + 30*24*3600;
            foreach ($store as &$item){
                if ($item['token'] === $hash){
                    $item['expires'] = $newExp;
                    break;
                }
            }
            _remember_store_write($store_path, $store);
            // refresh cookie
            $secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on');
            $params = [
                'expires'  => $newExp,
                'path'     => '/',
                'secure'   => $secure,
                'httponly' => true,
                'samesite' => 'Lax',
            ];
            setcookie($cookie_name, $raw, $params);
        }
        return; // authenticated
    }
}
// No valid session nor valid cookie → redirect
header('Location: login.php');
exit;
