<?php
/**
 * admin/admin_transactions.php
 * Liste unifiée des transactions (wallet, dépôts réels)
 * - Style & structure calqués sur admin/clients.php
 * - Recherche + pagination DataTables côté client
 * - Avatars/username via telegram_users
 * - EUR pour btc_deposits = calcul indicatif
 */
session_start();
require_once '../includes/db.php';
require_once __DIR__ . '/auth_guard.php';

if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
  header('Location: login.php');
  exit();
}

/** ===== Helpers similaires à clients.php ===== */
function btc_eur_rate(): float {
  $cache = __DIR__ . '/.btc_eur.cache.json';
  $now = time();

  if (is_file($cache) && ($now - filemtime($cache) < 60)) {
    $j = json_decode(@file_get_contents($cache), true);
    if (isset($j['eur']) && $j['eur'] > 0) return (float)$j['eur'];
  }

  $eur = 0.0;
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
  $local = @file_get_contents($scheme.'://'.$host.'/api/rates.php');
  if ($local) {
    $js = json_decode($local, true);
    if (!empty($js['ok']) && !empty($js['btc_eur'])) $eur = (float)$js['btc_eur'];
  }
  if ($eur <= 0) {
    $co = @file_get_contents('https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=eur');
    if ($co) {
      $js = json_decode($co, true);
      $eur = (float)($js['bitcoin']['eur'] ?? 0);
    }
  }
  if ($eur > 0) @file_put_contents($cache, json_encode(['eur'=>$eur]));
  return max(0.0, $eur);
}
function sats_to_btc($sats): float { return ((int)$sats)/1e8; }

$eurRate = btc_eur_rate();

/** ===== Récupération des transactions (UNION wallet + dépôts) ===== */
$q = trim($_GET['q'] ?? '');

$params = [];
$whereParts = [];
if ($q !== '') {
  $like = "%$q%";
  $whereParts[] = "(t.tx LIKE ? OR t.address LIKE ? OR u.username LIKE ? OR t.source LIKE ? OR t.direction LIKE ? OR CAST(t.telegram_id AS CHAR) LIKE ?)";
  array_push($params, $like, $like, $like, $like, $like, $like);
}
$whereSql = $whereParts ? ('WHERE '.implode(' AND ', $whereParts)) : '';

$union = "
  -- Transactions wallet (créditées), enrichies avec infos dépôt
  SELECT 'wallet' AS source,
         wt.id         AS rec_id,
         wt.telegram_id,
         wt.direction,
         wt.amount_sats AS sats,
         wt.eur_total   AS eur,
         wt.eur_per_btc AS rate_eurbtc,
         wt.ref         AS tx,
         d.address      AS address,
         d.confirmations,
         d.status       AS status,
         NULL           AS note,
         wt.created_at
  FROM wallet_tx wt
  LEFT JOIN btc_deposits d
         ON d.txid = wt.ref
        AND d.telegram_id = wt.telegram_id
  WHERE wt.direction = 'credit'

  UNION ALL

  -- Dépôts visibles uniquement si pas encore crédités dans wallet_tx
  SELECT 'deposit' AS source,
         d.id         AS rec_id,
         d.telegram_id,
         'credit'     AS direction,
         d.amount_sats AS sats,
         NULL          AS eur,
         NULL          AS rate_eurbtc,
         d.txid        AS tx,
         d.address     AS address,
         d.confirmations,
         d.status      AS status,
         NULL          AS note,
         d.created_at
  FROM btc_deposits d
  LEFT JOIN wallet_tx wt
         ON wt.ref = d.txid
        AND wt.telegram_id = d.telegram_id
        AND wt.direction = 'credit'
  WHERE wt.id IS NULL
";

$sql = "
  SELECT t.*, u.username, u.photo_url
  FROM ( $union ) t
  LEFT JOIN telegram_users u ON u.telegram_id = t.telegram_id
  $whereSql
  ORDER BY t.created_at DESC, t.rec_id DESC
";

$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

// Formatage pour affichage
$items = [];
foreach ($rows as $r) {
  $sats = (int)($r['sats'] ?? 0);
  $btc  = sats_to_btc($sats);

  $eur = null;
  if ($r['source'] === 'wallet' && isset($r['eur'])) {
    $eur = (float)$r['eur'];
  } elseif ($r['source'] === 'deposit' && $eurRate > 0) {
    $eur = $btc * $eurRate;
  }

  if ($r['source'] === 'wallet' && $r['direction'] === 'debit') {
    $btc = -abs($btc);
    if ($eur !== null) $eur = -abs($eur);
  }

  $items[] = [
    'avatar'        => $r['photo_url'] ?? '',
    'username'      => $r['username'] ?? '',
    'telegram_id'   => (string)($r['telegram_id'] ?? ''),
    'source'        => $r['source'],
    'direction'     => $r['direction'],
    'tx'            => $r['tx'] ?? '',
    'address'       => $r['address'] ?? '',
    'btc'           => $btc,
    'eur'           => $eur,
    'rate_eurbtc'   => isset($r['rate_eurbtc']) ? (float)$r['rate_eurbtc'] : null,
    'confirmations' => $r['confirmations'] ?? null,
    'status'        => $r['status'] ?? '',
    'note'          => $r['note'] ?? '',
    'created_at'    => $r['created_at'] ?? '',
  ];
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Transactions</title>

  <!-- Bootstrap 4 + AdminLTE (comme clients.php) -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">

  <!-- DataTables (version bootstrap4 pour matcher ton thème) -->
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap4.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css">
  <link rel="stylesheet" href="assets/responsive-fixes.css">

  <style>
    .table-responsive-wrap { overflow-x:auto; }
    .badge-id { background:#eef0f4; color:#333; border:1px solid #dfe3ea; font-weight:600; }
    .tx-hash { font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace; font-size:12px; }
    .avatar { width:36px; height:36px; border-radius:50%; object-fit:cover; }
    .muted { color:#6c757d; }
    .amount-neg { color:#c0392b; font-weight:600; }
    .amount-pos { color:#1e8449; font-weight:600; }
    .amount-zero { color:#6c757d; font-weight:600; }
    @media (max-width: 576px){
      td.actions > a { display:block; margin-bottom:6px; }
    }
  </style>
</head>
<body class="hold-transition sidebar-mini layout-fixed">
<div class="wrapper">

  <nav class="main-header navbar navbar-expand navbar-white navbar-light">
    <ul class="navbar-nav">
      <li class="nav-item">
        <a class="nav-link" data-widget="pushmenu" href="#" role="button">☰</a>
      </li>
      <li class="nav-item d-sm-none d-block">
        <span class="navbar-text">MeltLabz Admin</span>
      </li>
    </ul>
  </nav>

  <?php include 'sidebar.php'; ?>

  <div class="content-wrapper">
    <div class="content-header">
      <div class="container-fluid">
        <h1 class="m-0">💳 Transactions</h1>
        <p class="text-muted mb-0">
          Taux BTC→€ actuel (indicatif) :
          <strong><?= $eurRate>0 ? number_format($eurRate, 2, ',', ' ') . ' € / BTC' : '—' ?></strong>
        </p>

        <form class="form-inline mt-3" method="get" action="">
          <input type="text" class="form-control mr-2" name="q" value="<?= htmlspecialchars($q) ?>" placeholder="Recherche (tx, @user, id, address, type…)">
          <button class="btn btn-primary" type="submit">Rechercher</button>
        </form>
      </div>
    </div>

    <div class="content">
      <div class="container-fluid">
        <div class="card">
          <div class="card-body table-responsive-wrap">
            <table id="txTable" class="table table-bordered table-hover">
              <thead>
                <tr>
                  <th>Avatar</th>
                  <th>@Telegram</th>
                  <th>Type</th>
                  <th>TX / Réf.</th>
                  <th>Adresse</th>
                  <th>BTC</th>
                  <th>€</th>
                  <th>Confirm.</th>
                  <th>Statut</th>
                  <th>Date</th>
                </tr>
              </thead>
              <tbody>
              <?php foreach ($items as $it):
                $username = $it['username'] ? '@'.htmlspecialchars($it['username']) : '—';
                $id = htmlspecialchars($it['telegram_id']);
                $btc = (float)$it['btc'];
                $eur = $it['eur'];
                $cls = $btc < 0 ? 'amount-neg' : ($btc > 0 ? 'amount-pos' : 'amount-zero');
              ?>
                <tr>
                  <td><img src="<?= htmlspecialchars($it['avatar']) ?>" alt="avatar" class="avatar"></td>
                  <td><?= $username ?><br><span class="badge badge-id">ID: <?= $id ?></span></td>
                  <td><?= htmlspecialchars($it['source']) ?> (<?= htmlspecialchars($it['direction']) ?>)</td>
                  <td class="tx-hash"><?= htmlspecialchars($it['tx'] ?: ($it['source']==='wallet' ? ($it['note'] ?: $it['status'] ?: $it['tx']) : '—')) ?></td>
                  <td class="tx-hash"><?= htmlspecialchars($it['address'] ?: '—') ?></td>
                  <td class="<?= $cls ?>"><?= number_format($btc, 8, '.', '') ?></td>
                  <td>
                    <?php if ($eur !== null): ?>
                      <strong><?= number_format($eur, 2, ',', ' ') ?> €</strong>
                      <?php if ($it['source']==='deposit'): ?><span class="muted">~</span><?php endif; ?>
                    <?php else: ?>
                      —
                    <?php endif; ?>
                  </td>
                  <td><?= $it['confirmations'] !== null ? (int)$it['confirmations'] : '—' ?></td>
                  <td><?= htmlspecialchars($it['status'] ?: '—') ?></td>
                  <td><?= htmlspecialchars($it['created_at']) ?></td>
                </tr>
              <?php endforeach; ?>
              </tbody>
            </table>
          </div>
        </div>
      </div>
    </div>

  </div>
</div>

<!-- jQuery + Bootstrap + AdminLTE -->
<script src="https://cdn.jsdelivr.net/npm/jquery@3.6.4/dist/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.6.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/js/adminlte.min.js"></script>

<!-- DataTables (JS bootstrap4) -->
<script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap4.min.js"></script>
<script>
  $(function () {
    $('#txTable').DataTable({
      language: { url: '//cdn.datatables.net/plug-ins/1.13.6/i18n/fr-FR.json' },
      pageLength: 10,
      lengthMenu: [ [10, 20, 50, 100], [10, 20, 50, 100] ],
      order: [[ 9, 'desc' ]], // date
      responsive: false
    });
  });
</script>
</body>
</html>
